from machine import Pin, I2C, freq
from ssd1306 import SSD1306_I2C
from steptime import STEPTIME
import time
import math

freq(250_000_000)

# OLED
i2c = I2C(1, sda=Pin(6), scl=Pin(7), freq=200000)
oled = SSD1306_I2C(128, 64, i2c, addr=0x3C)

# Touch pads: [26,27,1,2,3,4]
PINS = [26, 27, 1, 2, 3, 4]  # speed+, speed-, zoom+, zoom-, shape, freeze
for p in PINS:
    Pin(p, Pin.IN, Pin.PULL_UP)

channels = [[STEPTIME(i, pin), [1e6]] for i, pin in enumerate(PINS)]
loop = 200
settle = 20000
touch_thresh = 12000

def read_touch():
    states = []
    for sm, minv in channels:
        sm.put(loop); sm.put(settle)
        r = 4294967296 - sm.get()
        if r < minv[0]:
            minv[0] = r
        states.append((r - minv[0]) > touch_thresh)
    return states

# Warm-up
t0 = time.ticks_ms()
while time.ticks_diff(time.ticks_ms(), t0) < 600:
    read_touch()
    time.sleep_ms(20)

W, H = 128, 64
HUD_H = 16
CX, CY = W // 2, (HUD_H + H) // 2

# Shapes: vertices + edges
SHAPES = []

# Cube
V_cube = [(-1,-1,-1),(1,-1,-1),(1,1,-1),(-1,1,-1), (-1,-1,1),(1,-1,1),(1,1,1),(-1,1,1)]
E_cube = [(0,1),(1,2),(2,3),(3,0),(4,5),(5,6),(6,7),(7,4),(0,4),(1,5),(2,6),(3,7)]
SHAPES.append(("CUBE", V_cube, E_cube))

# Pyramid (square base)
V_pyr = [(-1,-1,-1),(1,-1,-1),(1,1,-1),(-1,1,-1),(0,0,1.2)]
E_pyr = [(0,1),(1,2),(2,3),(3,0),(0,4),(1,4),(2,4),(3,4)]
SHAPES.append(("PYR", V_pyr, E_pyr))

# Octahedron
V_oct = [(1,0,0),(-1,0,0),(0,1,0),(0,-1,0),(0,0,1),(0,0,-1)]
E_oct = [(0,2),(0,3),(0,4),(0,5),(1,2),(1,3),(1,4),(1,5),(2,4),(2,5),(3,4),(3,5)]
SHAPES.append(("OCTA", V_oct, E_oct))

shape_i = 0

rot_speed = 0.06
zoom = 28.0
frozen = False
prev_freeze = False
prev_shape = False
ax = 0.0
ay = 0.0

def project(x, y, z):
    # simple perspective
    z += 3.5
    f = zoom / z
    sx = int(CX + x * f)
    sy = int(CY + y * f)
    return sx, sy

def line(x0, y0, x1, y1):
    # Bresenham line
    dx = abs(x1 - x0)
    sx = 1 if x0 < x1 else -1
    dy = -abs(y1 - y0)
    sy = 1 if y0 < y1 else -1
    err = dx + dy
    while True:
        if 0 <= x0 < W and HUD_H <= y0 < H:
            oled.pixel(x0, y0, 1)
        if x0 == x1 and y0 == y1:
            break
        e2 = 2 * err
        if e2 >= dy:
            err += dy
            x0 += sx
        if e2 <= dx:
            err += dx
            y0 += sy

while True:
    t = read_touch()

    # Controls
    if t[0]: rot_speed = min(0.25, rot_speed + 0.01)
    if t[1]: rot_speed = max(0.00, rot_speed - 0.01)
    if t[2]: zoom = min(60.0, zoom + 1.0)
    if t[3]: zoom = max(12.0, zoom - 1.0)

    # Shape cycle (edge detect)
    if t[4] and not prev_shape:
        shape_i = (shape_i + 1) % len(SHAPES)
    prev_shape = t[4]

    # Freeze toggle (edge detect)
    if t[5] and not prev_freeze:
        frozen = not frozen
    prev_freeze = t[5]

    if not frozen:
        ax += rot_speed * 0.7
        ay += rot_speed

    name, V, E = SHAPES[shape_i]

    # Precompute sin/cos
    c1, s1 = math.cos(ax), math.sin(ax)
    c2, s2 = math.cos(ay), math.sin(ay)

    # Transform + project verts
    pts = []
    for (x, y, z) in V:
        # rotate around X
        y1 = y * c1 - z * s1
        z1 = y * s1 + z * c1
        # rotate around Y
        x2 = x * c2 + z1 * s2
        z2 = -x * s2 + z1 * c2
        pts.append(project(x2, y1, z2))

    # Draw
    oled.fill(0)

    # HUD (yellow band)
    oled.fill_rect(0, 0, 128, HUD_H, 0)
    oled.text("WIRE", 0, 0, 1)
    oled.text(name, 48, 0, 1)
    oled.text("Z:%02d" % int(zoom), 92, 0, 1)
    if frozen:
        oled.text("FREEZE", 0, 8, 1)

    # Edges
    for a, b in E:
        x0, y0 = pts[a]
        x1, y1 = pts[b]
        line(x0, y0, x1, y1)

    oled.show()
    time.sleep_ms(20)
