from machine import Pin, I2C, freq
from ssd1306 import SSD1306_I2C
from steptime import STEPTIME
import time
import urandom
import math

freq(250_000_000)

# OLED
i2c = I2C(1, sda=Pin(6), scl=Pin(7), freq=200000)
oled = SSD1306_I2C(128, 64, i2c, addr=0x3C)

W, H = 128, 64
HUD_H = 16
Y0 = HUD_H
Y1 = H

# Touch pads: [26,27,1,2,3,4]
# 26 amp+, 27 amp-, 1 width+, 2 width-, 3 reseed, 4 freeze toggle
PINS = [26, 27, 1, 2, 3, 4]
for p in PINS:
    Pin(p, Pin.IN, Pin.PULL_UP)

channels = [[STEPTIME(i, pin), [1e6]] for i, pin in enumerate(PINS)]
loop = 200
settle = 20000
touch_thresh = 12000

def read_touch():
    states = []
    for sm, minv in channels:
        sm.put(loop); sm.put(settle)
        r = 4294967296 - sm.get()
        if r < minv[0]:
            minv[0] = r
        states.append((r - minv[0]) > touch_thresh)
    return states

# Warm-up baselines
t0 = time.ticks_ms()
while time.ticks_diff(time.ticks_ms(), t0) < 600:
    read_touch()
    time.sleep_ms(20)

# --- Parameters you can “play” ---
amp = 10          # wave height
width = 18        # peak width (bigger = broader pulses)
noise = 6         # jaggedness
speed = 0.08      # animation speed
frozen = False
prev_freeze = False
prev_seed = False

phase = 0.0
seed = 12345

def reseed():
    global seed
    seed = urandom.getrandbits(16) | 1

def rnd(n):
    # tiny deterministic-ish PRNG based on seed and n (fast, repeatable-ish)
    # returns -128..127
    v = (seed * (1103515245 + n*12345) + 12345) & 0xFFFFFFFF
    return ((v >> 16) & 255) - 128

reseed()

def hud():
    oled.fill_rect(0, 0, 128, HUD_H, 0)
    oled.text("WAVES", 0, 0, 1)
    oled.text("A:%02d" % amp, 52, 0, 1)
    oled.text("W:%02d" % width, 92, 0, 1)
    if frozen:
        oled.text("FREEZE", 0, 8, 1)

while True:
    t = read_touch()

    # controls (held)
    if t[0]: amp = min(22, amp + 1)
    if t[1]: amp = max(2, amp - 1)
    if t[2]: width = min(40, width + 1)
    if t[3]: width = max(6, width - 1)

    # reseed (edge)
    if t[4] and not prev_seed:
        reseed()
    prev_seed = t[4]

    # freeze toggle (edge)
    if t[5] and not prev_freeze:
        frozen = not frozen
    prev_freeze = t[5]

    if not frozen:
        phase += speed

    oled.fill(0)
    hud()

    # line stack settings
    lines = 20
    line_spacing = (Y1 - Y0 - 4) // lines
    cx = W // 2

    for li in range(lines):
        ybase = Y0 + 4 + li * line_spacing

        # peak envelope: strongest in middle lines
        mid = (lines - 1) / 2.0
        env_line = 1.0 - abs(li - mid) / mid
        env_line = max(0.0, env_line) ** 1.2

        # draw one polyline across the width
        prevy = ybase
        for x in range(W):
            dx = x - cx

            # peak envelope: strongest near center x, fades to edges
            env_x = math.exp(-(dx*dx) / (2.0 * width * width))

            # base wave + a sharper “pulse” component + roughness
            w = math.sin((x * 0.10) + phase + li*0.25)
            pulse = math.sin((x * 0.35) - phase*1.3 + li*0.12)
            rough = rnd(x + li*131) / 128.0

            yoff = (w * 0.6 + pulse * 0.4) * env_x
            yoff += rough * (noise / 10.0) * env_x

            y = int(ybase - (amp * env_line) * yoff)

            # clamp to drawing region
            if y < Y0: y = Y0
            if y >= H: y = H-1

            # draw line segment
            # (vertical stroke makes the “stacked contours” pop on 1-bit displays)
            if y < prevy:
                for yy in range(y, prevy+1):
                    oled.pixel(x, yy, 1)
            else:
                for yy in range(prevy, y+1):
                    oled.pixel(x, yy, 1)
            prevy = y

    oled.show()
    time.sleep_ms(30)
