from machine import Pin, I2C, freq
from ssd1306 import SSD1306_I2C
from steptime import STEPTIME
import time
import urandom

freq(250_000_000)

# OLED
i2c = I2C(1, sda=Pin(6), scl=Pin(7), freq=200000)
oled = SSD1306_I2C(128, 64, i2c, addr=0x3C)

# Touch pads: [26,27,1,2,3,4]
PINS = [26, 27, 1, 2, 3, 4]
for p in PINS:
    Pin(p, Pin.IN, Pin.PULL_UP)

channels = [[STEPTIME(i, pin), [1e6]] for i, pin in enumerate(PINS)]
loop = 200
settle = 20000
touch_thresh = 12000

def read_touch():
    states = []
    for sm, minv in channels:
        sm.put(loop); sm.put(settle)
        r = 4294967296 - sm.get()
        if r < minv[0]:
            minv[0] = r
        states.append((r - minv[0]) > touch_thresh)
    return states

# Warm-up
t0 = time.ticks_ms()
while time.ticks_diff(time.ticks_ms(), t0) < 600:
    read_touch()
    time.sleep_ms(20)

# Starfield state
W, H = 128, 64
HUD_H = 16
CX, CY = W // 2, (HUD_H + H) // 2

N = 80          # number of stars
speed = 3       # forward speed
spread = 64     # how wide stars spawn
warp = False
prev_warp = False
rand_flash = 0

# stars: x, y, z in "3D-ish" space
stars = []
def reset_star(i=None):
    x = (urandom.getrandbits(8) % (2*spread)) - spread
    y = (urandom.getrandbits(8) % (2*spread)) - spread
    z = (urandom.getrandbits(8) % 255) + 1
    if i is None:
        stars.append([x, y, z])
    else:
        stars[i][0], stars[i][1], stars[i][2] = x, y, z

for _ in range(N):
    reset_star()

while True:
    t = read_touch()

    # Controls:
    # 26 speed+, 27 speed-, 1 density+, 2 density-, 3 randomize, 4 warp toggle
    if t[0]: speed = min(12, speed + 1)
    if t[1]: speed = max(1, speed - 1)

    if t[2]:
        N = min(140, N + 5)
        while len(stars) < N:
            reset_star()
    if t[3]:
        N = max(20, N - 5)
        stars = stars[:N]

    if t[4]:
        # randomize / re-seed
        for i in range(len(stars)):
            reset_star(i)
        rand_flash = 6

    if t[5] and not prev_warp:
        warp = not warp
    prev_warp = t[5]

    # Draw
    oled.fill(0)

    # HUD
    oled.fill_rect(0, 0, 128, HUD_H, 0)
    oled.text("SPD:%02d" % speed, 0, 0, 1)
    oled.text("N:%03d" % N, 70, 0, 1)
    if warp:
        oled.text("WARP", 0, 8, 1)
    if rand_flash > 0:
        oled.text("RND", 70, 8, 1)
        rand_flash -= 1

    # Stars (project)
    # Using simple perspective: sx = cx + x*256/z, sy = cy + y*256/z
    # Keep it fast: integer math, and only plot if in bounds
    for i in range(len(stars)):
        x, y, z = stars[i]
        z -= speed if not warp else (speed * 3)
        if z <= 1:
            reset_star(i)
            continue
        stars[i][2] = z

        sx = CX + (x * 256) // z
        sy = CY + (y * 256) // z
        if 0 <= sx < W and HUD_H <= sy < H:
            oled.pixel(sx, sy, 1)
        else:
            # if it flew off screen, respawn it
            reset_star(i)

    oled.show()
    time.sleep_ms(20)
