from machine import Pin, I2C, freq
from ssd1306 import SSD1306_I2C
from steptime import STEPTIME
import time
import urandom

freq(250_000_000)

# OLED
i2c = I2C(1, sda=Pin(6), scl=Pin(7), freq=200000)
oled = SSD1306_I2C(128, 64, i2c, addr=0x3C)

# Touch pads
PINS = [26, 27, 1, 2, 3, 4]
for p in PINS:
    Pin(p, Pin.IN, Pin.PULL_UP)

channels = [[STEPTIME(i, pin), [1e6]] for i, pin in enumerate(PINS)]
loop = 200
settle = 20000
touch_thresh = 12000

def read_touch():
    states = []
    for sm, minv in channels:
        sm.put(loop); sm.put(settle)
        r = 4294967296 - sm.get()
        if r < minv[0]:
            minv[0] = r
        states.append((r - minv[0]) > touch_thresh)
    return states

# Warm-up
t0 = time.ticks_ms()
while time.ticks_diff(time.ticks_ms(), t0) < 600:
    read_touch()
    time.sleep_ms(20)

# Ripple sim on a smaller grid (fast), then scaled up
W, H = 64, 24          # simulation grid
HUD_H = 16             # keep top band clean
scale = 2              # 64*2=128 pixels wide
yoff = HUD_H           # draw below HUD
damping = 5            # 1..10 (higher = more damping / calmer)

# Two buffers: prev and cur height
prev = [[0]*W for _ in range(H)]
cur  = [[0]*W for _ in range(H)]

def splash(cx, cy, amp=80):
    if 1 <= cx < W-1 and 1 <= cy < H-1:
        cur[cy][cx] += amp
        cur[cy][cx-1] += amp//2
        cur[cy][cx+1] += amp//2
        cur[cy-1][cx] += amp//2
        cur[cy+1][cx] += amp//2

def clear():
    for y in range(H):
        rowp = prev[y]
        rowc = cur[y]
        for x in range(W):
            rowp[x] = 0
            rowc[x] = 0

prev_btn = [False]*6

while True:
    t = read_touch()

    # Edge detect
    pressed = [t[i] and not prev_btn[i] for i in range(6)]
    prev_btn = t

    # Controls
    if pressed[0]:  # left splash
        splash(W//4, H//2, 120)
    if pressed[1]:  # right splash
        splash(3*W//4, H//2, 120)
    if pressed[2]:  # center splash
        splash(W//2, H//2, 160)
    if pressed[3]:  # clear
        clear()
    if pressed[4]:  # more damping (calmer)
        damping = min(10, damping + 1)
    if pressed[5]:  # less damping (more lively)
        damping = max(1, damping - 1)

    # Step simulation (classic ripple)
    # new = (neighbors avg *2 - prev) ; then apply damping
    for y in range(1, H-1):
        for x in range(1, W-1):
            v = (
                cur[y-1][x] +
                cur[y+1][x] +
                cur[y][x-1] +
                cur[y][x+1]
            ) // 2 - prev[y][x]
            # damping: bring toward zero
            v -= v // damping
            prev[y][x] = cur[y][x]
            cur[y][x] = v

    # Render: threshold height into black/white contours
    oled.fill(0)

    # HUD
    oled.fill_rect(0, 0, 128, HUD_H, 0)
    oled.text("RIPPLES", 0, 0, 1)
    oled.text("DAMP:%d" % damping, 0, 8, 1)
    oled.text("L C R CLR", 70, 8, 1)

    # Draw ripples (simple contour-ish look)
    for y in range(H):
        py = yoff + y*scale
        row = cur[y]
        for x in range(W):
            h = row[x]
            # Contour threshold: tweak these for different looks
            on = (h > 10) or (h < -10)
            if on:
                px = x*scale
                oled.pixel(px, py, 1)
                oled.pixel(px+1, py, 1)
                oled.pixel(px, py+1, 1)
                oled.pixel(px+1, py+1, 1)

    oled.show()
    time.sleep_ms(20)
