from machine import Pin, I2C, freq
from ssd1306 import SSD1306_I2C
from steptime import STEPTIME
import time, math, urandom

freq(250_000_000)

# OLED
i2c = I2C(1, sda=Pin(6), scl=Pin(7), freq=200000)
oled = SSD1306_I2C(128, 64, i2c, addr=0x3C)

W, H = 128, 64
Y0, Y1 = 16, 64   # blue region

# Touch pads: index 4 (pin 3) redraw, index 5 (pin 4) invert toggle
PINS = [26, 27, 1, 2, 3, 4]
for p in PINS:
    Pin(p, Pin.IN, Pin.PULL_UP)

channels = [[STEPTIME(i, pin), [1e6]] for i, pin in enumerate(PINS)]
loop, settle = 200, 20000
touch_thresh = 12000

def read_touch():
    out = []
    for sm, m in channels:
        sm.put(loop); sm.put(settle)
        r = 4294967296 - sm.get()
        if r < m[0]:
            m[0] = r
        out.append((r - m[0]) > touch_thresh)
    return out

# warm-up
t0 = time.ticks_ms()
while time.ticks_diff(time.ticks_ms(), t0) < 600:
    read_touch()
    time.sleep_ms(20)

# Max-style helpers
def rand(minv, maxv):
    return (urandom.getrandbits(24) / 16777215) * (maxv - minv) + minv

def rand_int(minv, maxv):
    return minv + (urandom.getrandbits(16) % (maxv - minv + 1))

def rand_normal(mu=0.0, sigma=1.0):
    s = 0.0
    for _ in range(6):
        s += rand(-1.0, 1.0)
    return mu + sigma * (s / 6.0)

def normal_pdf(x, mu, sigma):
    s2 = sigma * sigma
    if s2 <= 1e-9:
        return 0.0
    return math.exp(-((x - mu) * (x - mu)) / (2.0 * s2)) / math.sqrt(2.0 * math.pi * s2)

invert = False

def draw_vertical(seed=None):
    if seed is not None:
        try:
            urandom.seed(seed)
        except:
            pass

    oled.fill(0)

    # --- Margins: not to edges, and not starting at "left/top" ---
    y_min = Y0 + 4
    y_max = Y1 - 4

    x_left_margin  = 18     # keeps baseline away from left edge ("top" in your mental rotation)
    x_right_margin = 10
    x_min = x_left_margin
    x_max = W - x_right_margin

    # --- Fewer lines, more spaced ---
    col_step = 9            # bigger = fewer columns (try 10–12 for even fewer)
    n_cols = max(4, (x_max - x_min) // col_step)

    # points along each column
    my = (y_min + y_max) / 2.0

    # --- Lower peaks / calmer ---
    scale = 260.0           # lower = less high peaks
    noise_gain = 60.0       # lower = cleaner
    smooth_a = 0.35         # higher = smoother

    x0 = x_min
    for i in range(n_cols):
        # 2..3 modes for simpler shapes
        n_modes = rand_int(2, 3)
        mus, sigmas = [], []
        for _ in range(n_modes):
            mus.append(rand_normal(my, 7.0))
            sigmas.append(abs(rand_normal(6.5, 3.0)) + 2.0)

        w = x0
        prev_x = x0

        for y in range(y_min, y_max):
            m = 0.0
            for k in range(n_modes):
                m += normal_pdf(y, mus[k], sigmas[k])

            # baseline is x0; displacement left by m (but bounded by x_min)
            xx_raw = (x0 - scale * m) + (m * rand(0.0, noise_gain)) + rand(0.0, 0.4)
            xx = smooth_a * w + (1.0 - smooth_a) * xx_raw
            w = xx

            xi = int(xx)
            if xi < x_min:
                xi = x_min
            if xi > x_max:
                xi = x_max

            # draw connected horizontal segment at this y
            if xi < prev_x:
                for xx2 in range(xi, prev_x + 1):
                    oled.pixel(xx2, y, 0 if invert else 1)
            else:
                for xx2 in range(prev_x, xi + 1):
                    oled.pixel(xx2, y, 0 if invert else 1)

            prev_x = xi

        x0 += col_step
        if x0 > x_max:
            break

    oled.show()

# initial render
draw_vertical(seed=urandom.getrandbits(16))

# touch loop
prev = [False]*6
while True:
    t = read_touch()
    pressed = [t[i] and not prev[i] for i in range(6)]
    prev = t

    if pressed[4]:  # pin 3 -> redraw
        draw_vertical(seed=urandom.getrandbits(16))

    if pressed[5]:  # pin 4 -> invert toggle
        invert = not invert
        draw_vertical(seed=urandom.getrandbits(16))

    time.sleep_ms(60)
