from machine import Pin, I2C, freq
from ssd1306 import SSD1306_I2C
from steptime import STEPTIME
import time
import math
import urandom

freq(250_000_000)

# OLED
i2c = I2C(1, sda=Pin(6), scl=Pin(7), freq=200000)
oled = SSD1306_I2C(128, 64, i2c, addr=0x3C)

# Touch pads: [26,27,1,2,3,4]
# 26 spacing+, 27 spacing-, 1 rot speed+, 2 rot speed-, 3 invert, 4 freeze/reset
PINS = [26, 27, 1, 2, 3, 4]
for p in PINS:
    Pin(p, Pin.IN, Pin.PULL_UP)

channels = [[STEPTIME(i, pin), [1e6]] for i, pin in enumerate(PINS)]
loop = 200
settle = 20000
touch_thresh = 12000

def read_touch():
    states = []
    for sm, minv in channels:
        sm.put(loop); sm.put(settle)
        r = 4294967296 - sm.get()
        if r < minv[0]:
            minv[0] = r
        states.append((r - minv[0]) > touch_thresh)
    return states

# Warm-up
t0 = time.ticks_ms()
while time.ticks_diff(time.ticks_ms(), t0) < 600:
    read_touch()
    time.sleep_ms(20)

W, H = 128, 64
HUD_H = 16
CX, CY = W // 2, (HUD_H + H) // 2

# Parameters
spacing = 8            # 3..24 (lower = tighter stripes)
rot_speed = 0.04       # radians per frame
warp = 6               # 0..20 (higher = more bend)
invert = False
frozen = False

prev = [False]*6
phase = 0.0
rand_flash = 0

def hud():
    oled.fill_rect(0, 0, 128, HUD_H, 0)
    oled.text("MOIRE", 0, 0, 1)
    oled.text("S:%02d" % spacing, 48, 0, 1)
    oled.text("W:%02d" % warp, 92, 0, 1)
    if frozen:
        oled.text("FREEZE", 0, 8, 1)
    if invert:
        oled.text("INV", 92, 8, 1)
    if rand_flash > 0:
        oled.text("RND", 48, 8, 1)

while True:
    t = read_touch()
    pressed = [t[i] and not prev[i] for i in range(6)]
    prev = t

    # Controls:
    # 26 spacing+, 27 spacing-, 1 rot+, 2 rot-, 3 invert toggle, 4 freeze toggle (long press also randomize)
    if t[0]:
        spacing = min(24, spacing + 1)
    if t[1]:
        spacing = max(3, spacing - 1)
    if t[2]:
        rot_speed = min(0.20, rot_speed + 0.005)
    if t[3]:
        rot_speed = max(0.00, rot_speed - 0.005)

    if pressed[4]:
        invert = not invert

    if pressed[5]:
        frozen = not frozen

    # Long press on pad 4 (pin 4) randomizes warp
    if t[5] and not frozen:
        # if you keep holding, slowly drift warp for fun
        warp = 2 + (urandom.getrandbits(5) % 18)
        rand_flash = 6

    if not frozen:
        phase += rot_speed
        if rand_flash > 0:
            rand_flash -= 1

    # Draw (2x2 blocks for speed; still looks sharp)
    oled.fill(0)
    hud()

    c = math.cos(phase)
    s = math.sin(phase)

    # Threshold for stripes:
    # Use rotated coords + radial warp + sine -> black/white bands.
    for y in range(HUD_H, H, 2):
        yy = y - CY
        for x in range(0, W, 2):
            xx = x - CX

            # Rotate
            xr = xx * c - yy * s
            yr = xx * s + yy * c

            # Radial warp (cheap “lens”)
            r = (xx*xx + yy*yy) ** 0.5
            v = xr + warp * math.sin((r / 8.0) + phase)

            # Stripe pattern
            band = int(v / spacing)

            on = (band & 1) == 0
            if invert:
                on = not on

            if on:
                oled.pixel(x, y, 1)
                if x+1 < W: oled.pixel(x+1, y, 1)
                if y+1 < H: oled.pixel(x, y+1, 1)
                if x+1 < W and y+1 < H: oled.pixel(x+1, y+1, 1)

    oled.show()
    time.sleep_ms(20)
