from machine import Pin, I2C, freq
from ssd1306 import SSD1306_I2C
from steptime import STEPTIME
import time
import urandom

freq(250_000_000)

# OLED
i2c = I2C(1, sda=Pin(6), scl=Pin(7), freq=200000)
oled = SSD1306_I2C(128, 64, i2c, addr=0x3C)

oled.fill(0)
oled.text("metaballs...", 0, 0, 1)
oled.show()

# Touch pads
PINS = [26, 27, 1, 2, 3, 4]  # speed+, speed-, size+, size-, random, freeze
for p in PINS:
    Pin(p, Pin.IN, Pin.PULL_UP)

channels = [[STEPTIME(i, pin), [1e6]] for i, pin in enumerate(PINS)]
loop = 200
settle = 20000
touch_thresh = 12000

def read_touch():
    states = []
    for sm, minv in channels:
        sm.put(loop); sm.put(settle)
        r = 4294967296 - sm.get()
        if r < minv[0]:
            minv[0] = r
        states.append((r - minv[0]) > touch_thresh)
    return states

# Warmup
t0 = time.ticks_ms()
while time.ticks_diff(time.ticks_ms(), t0) < 600:
    read_touch()
    time.sleep_ms(20)

# Metaballs
balls = [
    [20, 22,  1,  1],
    [90, 44, -1,  1],
    [60, 58,  1, -1],
]

speed = 2          # 1..6
size = 900         # 200..1600 (strength)
threshold = 55     # 25..120 (higher = thinner)
frozen = False
prev_freeze = False
rand_flash = 0

def reset_balls():
    global balls
    balls = []
    for _ in range(3):
        x = urandom.getrandbits(7) % 128
        y = 16 + (urandom.getrandbits(6) % (64-16))
        dx = 1 if (urandom.getrandbits(1) == 0) else -1
        dy = 1 if (urandom.getrandbits(1) == 0) else -1
        balls.append([x, y, dx, dy])

reset_balls()

while True:
    touch = read_touch()

    if touch[0]: speed = min(6, speed + 1)
    if touch[1]: speed = max(1, speed - 1)
    if touch[2]: size = min(1600, size + 80)
    if touch[3]: size = max(200, size - 80)

    if touch[4]:
        reset_balls()
        rand_flash = 6

    if touch[5] and not prev_freeze:
        frozen = not frozen
    prev_freeze = touch[5]

    if not frozen:
        for b in balls:
            b[0] += b[2] * speed
            b[1] += b[3] * speed
            if b[0] <= 0 or b[0] >= 127:
                b[2] = -b[2]
                b[0] = max(0, min(127, b[0]))
            if b[1] <= 16 or b[1] >= 63:
                b[3] = -b[3]
                b[1] = max(16, min(63, b[1]))

    oled.fill(0)

    # HUD (top 16px)
    oled.fill_rect(0, 0, 128, 16, 0)
    oled.text("SPD:%d" % speed, 0, 0, 1)
    oled.text("SZ:%d" % size, 56, 0, 1)
    if frozen:
        oled.text("FREEZE", 0, 8, 1)
    if rand_flash > 0:
        oled.text("RAND!", 70, 8, 1)
        rand_flash -= 1

    # Draw in 2x2 blocks for speed (x,y step 2)
    for y in range(16, 64, 2):
        for x in range(0, 128, 2):
            acc = 0
            for bx, by, _, _ in balls:
                dx = x - bx
                dy = y - by
                acc += size // (dx*dx + dy*dy + 1)

            if acc > threshold:
                oled.pixel(x, y, 1)
                if x+1 < 128: oled.pixel(x+1, y, 1)
                if y+1 < 64:  oled.pixel(x, y+1, 1)
                if x+1 < 128 and y+1 < 64: oled.pixel(x+1, y+1, 1)

    oled.show()
    time.sleep_ms(20)
