# lavalamp.py
# Simple metaball lava lamp for 128x64 SSD1306

from machine import Pin, I2C, freq
from ssd1306 import SSD1306_I2C
import math
import time
import urandom

freq(250_000_000)

# ---- Display ----
W, H = 128, 64
i2c = I2C(1, sda=Pin(6), scl=Pin(7), freq=200000)
oled = SSD1306_I2C(W, H, i2c, addr=0x3C)

# ---- Lava blobs ----
NUM_BLOBS = 4

class Blob:
    def __init__(self):
        self.x = urandom.getrandbits(7) % W
        self.y = urandom.getrandbits(6) % H
        self.r = 10 + (urandom.getrandbits(4) % 8)
        self.vx = (urandom.getrandbits(2) - 1) * 0.4
        self.vy = ((urandom.getrandbits(3) % 3) - 1) * 0.5

    def move(self):
        self.x += self.vx
        self.y += self.vy

        if self.x < 0 or self.x > W:
            self.vx *= -1
        if self.y < 0 or self.y > H:
            self.vy *= -1

blobs = [Blob() for _ in range(NUM_BLOBS)]

THRESHOLD = 1.0

while True:
    oled.fill(0)

    # Move blobs
    for b in blobs:
        b.move()

    # Metaball field
    for y in range(0, H, 2):      # skip every other row for speed
        for x in range(0, W, 2):  # skip every other column
            field = 0
            for b in blobs:
                dx = x - b.x
                dy = y - b.y
                dist2 = dx*dx + dy*dy
                if dist2 == 0:
                    dist2 = 1
                field += (b.r * b.r) / dist2

            if field > THRESHOLD:
                # draw 2x2 block to compensate for skipping
                oled.fill_rect(x, y, 2, 2, 1)

    oled.show()
    time.sleep_ms(30)
