from machine import Pin, I2C, freq
from ssd1306 import SSD1306_I2C
from steptime import STEPTIME
import time
import urandom

freq(250_000_000)

# --- OLED ---
i2c = I2C(1, sda=Pin(6), scl=Pin(7), freq=200000)
oled = SSD1306_I2C(128, 64, i2c, addr=0x3C)

W, H = 128, 64
HUD_H = 16

# --- Touch pads ---
PINS = [26, 27, 1, 2, 3, 4]  # density+, density-, thick+, thick-, reseed, freeze
for p in PINS:
    Pin(p, Pin.IN, Pin.PULL_UP)

channels = [[STEPTIME(i, pin), [1e6]] for i, pin in enumerate(PINS)]
loop = 200
settle = 20000
touch_thresh = 12000

def read_touch():
    states = []
    for sm, minv in channels:
        sm.put(loop); sm.put(settle)
        r = 4294967296 - sm.get()
        if r < minv[0]:
            minv[0] = r
        states.append((r - minv[0]) > touch_thresh)
    return states

# Warm-up baselines
t0 = time.ticks_ms()
while time.ticks_diff(time.ticks_ms(), t0) < 600:
    read_touch()
    time.sleep_ms(20)

# --- Drawing helpers (1-bit "line weight" via offsets) ---
def hline_thick(x0, x1, y, t=1):
    if x0 > x1: x0, x1 = x1, x0
    for dy in range(-(t//2), t - (t//2)):
        yy = y + dy
        if HUD_H <= yy < H:
            oled.hline(max(0, x0), yy, min(W, x1+1) - max(0, x0), 1)

def vline_thick(x, y0, y1, t=1):
    if y0 > y1: y0, y1 = y1, y0
    for dx in range(-(t//2), t - (t//2)):
        xx = x + dx
        if 0 <= xx < W:
            yy0 = max(HUD_H, y0)
            yy1 = min(H-1, y1)
            if yy0 <= yy1:
                oled.vline(xx, yy0, yy1 - yy0 + 1, 1)

def rect_frame(x0, y0, x1, y1, t=1, gaps=0, jitter=0):
    # optionally draw broken edges by skipping segments
    if x0 > x1: x0, x1 = x1, x0
    if y0 > y1: y0, y1 = y1, y0
    y0 = max(HUD_H, y0)
    y1 = min(H-1, y1)
    if y0 >= y1 or x0 >= x1:
        return

    def segs(length, chunks):
        # returns list of (a,b) segments to draw along an edge
        if chunks <= 0:
            return [(0, length)]
        out = []
        pos = 0
        for _ in range(chunks):
            step = max(2, length // (chunks + 1))
            on = max(2, step - (urandom.getrandbits(3) % (step//2 + 1)))
            off = max(1, step - on)
            if pos + on > length:
                on = length - pos
            out.append((pos, pos + on))
            pos += on + off
            if pos >= length:
                break
        return out

    # top
    Lx = x1 - x0
    for a,b in segs(Lx, gaps):
        j = (urandom.getrandbits(4) % (2*jitter+1)) - jitter if jitter else 0
        hline_thick(x0 + a, x0 + b, y0 + j, t)
    # bottom
    for a,b in segs(Lx, gaps):
        j = (urandom.getrandbits(4) % (2*jitter+1)) - jitter if jitter else 0
        hline_thick(x0 + a, x0 + b, y1 + j, t)
    # left
    Ly = y1 - y0
    for a,b in segs(Ly, gaps):
        j = (urandom.getrandbits(4) % (2*jitter+1)) - jitter if jitter else 0
        vline_thick(x0 + j, y0 + a, y0 + b, t)
    # right
    for a,b in segs(Ly, gaps):
        j = (urandom.getrandbits(4) % (2*jitter+1)) - jitter if jitter else 0
        vline_thick(x1 + j, y0 + a, y0 + b, t)

def hud(density, thick, frozen, seed_tag):
    oled.fill_rect(0, 0, 128, HUD_H, 0)
    oled.text("GLITCH", 0, 0, 1)
    oled.text("N:%02d" % density, 64, 0, 1)
    oled.text("T:%d" % thick, 104, 0, 1)
    if frozen:
        oled.text("FREEZE", 0, 8, 1)
    oled.text(seed_tag, 80, 8, 1)

# --- Composition parameters ---
density = 10     # number of boxes/frames
thick = 2        # 1..4
frozen = False
prev_freeze = False
prev_reseed = False
seed_tag = "S0"
tick = 0

def reseed():
    # just for HUD feedback
    nonlocal_vars[0] = "S%X" % (urandom.getrandbits(8) & 0xF)

# hacky: allow reseed() to update seed_tag without global statement spam
nonlocal_vars = [seed_tag]

def draw_scene():
    oled.fill(0)
    hud(density, thick, frozen, nonlocal_vars[0])

    # big outer border, slightly broken
    rect_frame(2, HUD_H+2, W-3, H-3, t=thick, gaps=6, jitter=1)

    # a second “misregistered” border copy (printer offset vibe)
    offx = ((urandom.getrandbits(3) % 3) - 1)
    offy = ((urandom.getrandbits(3) % 3) - 1)
    rect_frame(2+offx, HUD_H+2+offy, W-3+offx, H-3+offy, t=1, gaps=10, jitter=0)

    # internal boxes in a loose grid
    cols = 4
    rows = 3
    gw = W // cols
    gh = (H - HUD_H) // rows

    # draw random “frames” with variable weight + glitches
    for _ in range(density):
        c = urandom.getrandbits(8) % cols
        r = urandom.getrandbits(8) % rows

        x0 = c * gw + (urandom.getrandbits(6) % (gw//3 + 1))
        y0 = HUD_H + r * gh + (urandom.getrandbits(6) % (gh//3 + 1))
        x1 = (c+1)*gw - 2 - (urandom.getrandbits(6) % (gw//3 + 1))
        y1 = HUD_H + (r+1)*gh - 2 - (urandom.getrandbits(6) % (gh//3 + 1))

        # occasional skinny/thick contrast
        t = thick if (urandom.getrandbits(1) == 0) else 1

        # glitch params
        gaps = 2 + (urandom.getrandbits(3) % 7)
        jit  = (urandom.getrandbits(2) % 3)

        rect_frame(x0, y0, x1, y1, t=t, gaps=gaps, jitter=jit)

        # “scanline cut” glitch: slice a rectangle edge with a horizontal bar
        if (urandom.getrandbits(3) == 0):
            yy = y0 + (urandom.getrandbits(6) % max(1, (y1-y0)))
            hline_thick(x0, x1, yy, t=1)

        # “registration shift”: redraw a fragment offset
        if (urandom.getrandbits(3) == 1):
            dx = (urandom.getrandbits(3) % 5) - 2
            dy = (urandom.getrandbits(3) % 5) - 2
            rect_frame(x0+dx, y0+dy, x1+dx, y1+dy, t=1, gaps=gaps+3, jitter=0)

    # occasional vertical “tear”
    if (urandom.getrandbits(2) == 0):
        x = 4 + (urandom.getrandbits(7) % (W-8))
        vline_thick(x, HUD_H+2, H-3, t=1)

    oled.show()

# --- Main loop ---
while True:
    t = read_touch()
    pressed_freeze = t[5] and not prev_freeze
    prev_freeze = t[5]

    # Density adjust (held)
    if t[0]: density = min(24, density + 1)
    if t[1]: density = max(2, density - 1)

    # Thickness adjust (held)
    if t[2]: thick = min(4, thick + 1)
    if t[3]: thick = max(1, thick - 1)

    # Reseed (edge)
    if t[4] and not prev_reseed:
        nonlocal_vars[0] = "S%X" % (urandom.getrandbits(8) & 0xF)
        # immediate redraw
        draw_scene()
    prev_reseed = t[4]

    if pressed_freeze:
        frozen = not frozen

    # animate by redrawing every few ticks (keeps it “glitchy”)
    if not frozen:
        tick += 1
        if (tick % 6) == 0:
            draw_scene()

    time.sleep_ms(30)
