from machine import Pin, I2C, freq
from ssd1306 import SSD1306_I2C
from steptime import STEPTIME
import time
import urandom

freq(250_000_000)

# --- OLED ---
i2c = I2C(1, sda=Pin(6), scl=Pin(7), freq=200000)
oled = SSD1306_I2C(128, 64, i2c, addr=0x3C)

# --- Touch pads: [26,27,1,2,3,4] ---
PINS = [26, 27, 1, 2, 3, 4]  # speed+, speed-, density+, density-, reset, freeze
for p in PINS:
    Pin(p, Pin.IN, Pin.PULL_UP)

channels = [[STEPTIME(i, pin), [1e6]] for i, pin in enumerate(PINS)]
loop = 200
settle = 20000
touch_thresh = 12000  # adjust if touch too sensitive / not sensitive

def read_touch():
    states = []
    for sm, minv in channels:
        sm.put(loop); sm.put(settle)
        r = 4294967296 - sm.get()
        if r < minv[0]:
            minv[0] = r
        states.append((r - minv[0]) > touch_thresh)
    return states

# Warm-up baselines
t0 = time.ticks_ms()
while time.ticks_diff(time.ticks_ms(), t0) < 600:
    read_touch()
    time.sleep_ms(20)

# --- Precompute sin table (fast-ish "flow") ---
# values in [-127..127]
sin256 = bytearray(256)
import math
for i in range(256):
    sin256[i] = int(127 * math.sin(i * (2 * math.pi / 256))) & 0xFF

def s8(v):  # convert 0..255 byte to signed int8
    return v - 256 if v > 127 else v

# 16-direction unit vectors (dx,dy)
DIRS = [
    ( 1,  0), ( 1,  1), ( 0,  1), (-1,  1),
    (-1,  0), (-1, -1), ( 0, -1), ( 1, -1),
    ( 2,  0), ( 2,  1), ( 1,  2), ( 0,  2),
    (-1,  2), (-2,  1), (-2,  0), (-2, -1),
]

W, H = 128, 64
HUD_H = 16

# Particles
max_particles = 140
particles = []

def reseed(n):
    particles.clear()
    for _ in range(n):
        x = urandom.getrandbits(7) % W
        y = HUD_H + (urandom.getrandbits(6) % (H - HUD_H))
        particles.append([x, y])

# Initial params
speed = 1          # particle steps per frame (1..5)
density = 90       # number of particles (20..140)
drift = 2          # field drift speed (1..8)
frozen = False
prev_freeze = False
rand_flash = 0

reseed(density)

# Helps the “silky” look: only clear every few frames so trails remain
clear_every = 4
frame = 0

while True:
    t = read_touch()

    # Edge-detect for toggles
    pressed_freeze = t[5] and not prev_freeze
    prev_freeze = t[5]

    # Controls:
    # 26 speed+, 27 speed-, 1 density+, 2 density-, 3 reseed, 4 freeze toggle
    if t[0]: speed = min(5, speed + 1)
    if t[1]: speed = max(1, speed - 1)

    if t[2]:
        density = min(max_particles, density + 5)
        while len(particles) < density:
            x = urandom.getrandbits(7) % W
            y = HUD_H + (urandom.getrandbits(6) % (H - HUD_H))
            particles.append([x, y])
    if t[3]:
        density = max(20, density - 5)
        if len(particles) > density:
            del particles[density:]

    if t[4]:
        reseed(density)
        rand_flash = 6

    if pressed_freeze:
        frozen = not frozen

    # Field time
    if not frozen:
        frame += drift

    # Trails: don’t clear every frame
    if (frame % clear_every) == 0:
        oled.fill(0)

    # HUD (always redraw cleanly)
    oled.fill_rect(0, 0, 128, HUD_H, 0)
    oled.text("FLOW", 0, 0, 1)
    oled.text("SPD:%d" % speed, 40, 0, 1)
    oled.text("N:%03d" % density, 92, 0, 1)
    if frozen:
        oled.text("FREEZE", 0, 8, 1)
    if rand_flash > 0:
        oled.text("RND", 92, 8, 1)
        rand_flash -= 1

    # Flow field step:
    # Use a cheap “noise-ish” angle from sin tables
    # angle_index = f(x,y,t) -> pick one of 16 directions
    for p in particles:
        x, y = p[0], p[1]

        for _ in range(speed):
            # sample the field
            # (tweak the multipliers for different "silk" character)
            a = (x * 3 + frame) & 255
            b = (y * 5 + (frame >> 1)) & 255
            c = (x * 2 + y * 2 + (frame >> 2)) & 255

            vx = s8(sin256[a]) + s8(sin256[c])
            vy = s8(sin256[b]) - s8(sin256[c])

            # turn vector into a 0..15 direction bucket
            ang = (vx + (vy << 1)) & 255
            dx, dy = DIRS[ang >> 4]

            x += dx
            y += dy

            # wrap (keep below HUD)
            if x < 0: x += W
            elif x >= W: x -= W
            if y < HUD_H: y = H - 1
            elif y >= H: y = HUD_H

            oled.pixel(x, y, 1)

        p[0], p[1] = x, y

    oled.show()
    time.sleep_ms(20)
