from machine import Pin, ADC, PWM
import time

FWD_PIN  = 27
BWD_PIN  = 26
STOP_PIN = 22

POT_PIN  = 28  

PWM_PIN  = 2
IN2_PIN  = 3
IN1_PIN  = 4
STBY_PIN = 5


btn_fwd  = Pin(FWD_PIN,  Pin.IN, Pin.PULL_UP)
btn_bwd  = Pin(BWD_PIN,  Pin.IN, Pin.PULL_UP)
btn_stop = Pin(STOP_PIN, Pin.IN, Pin.PULL_UP)


pot = ADC(POT_PIN)


in1  = Pin(IN1_PIN, Pin.OUT)
in2  = Pin(IN2_PIN, Pin.OUT)
stby = Pin(STBY_PIN, Pin.OUT)

pwm = PWM(Pin(PWM_PIN))
pwm.freq(1000)


state = 0


last_fwd  = 1
last_bwd  = 1
last_stop = 1


DEBOUNCE_MS = 30
last_fwd_change  = time.ticks_ms()
last_bwd_change  = time.ticks_ms()
last_stop_change = time.ticks_ms()


DEADBAND = 2000  


last_print_ms = time.ticks_ms()
PRINT_EVERY_MS = 200

def state_text(s):
    if s == 0:
        return "STOP"
    if s == 1:
        return "FWD"
    if s == 2:
        return "BWD"
    return "UNK"

print("Sistema listo (MicroPython, PULL_UP).")

def pressed_edge(read_now, last_read, last_change_ms):
    """
    Detecta "presión" con PULL_UP:
      - reposo 1
      - presionado 0
    Flanco de presión = 1 -> 0
    Incluye debounce básico.
    Retorna: (edge, new_last_read, new_last_change_ms)
    """
    edge = 0
    now_ms = time.ticks_ms()

    
    if read_now != last_read:
        if time.ticks_diff(now_ms, last_change_ms) >= DEBOUNCE_MS:
            last_change_ms = now_ms
            
            if last_read == 1 and read_now == 0:
                edge = 1
            last_read = read_now

    return edge, last_read, last_change_ms

while True:
    
    raw = pot.read_u16()          
    volts = raw * 3.3 / 65535
    duty = raw

    if duty < DEADBAND:
        duty = 0

    f = btn_fwd.value()
    b = btn_bwd.value()
    s = btn_stop.value()

    
    fwd_edge, last_fwd, last_fwd_change = pressed_edge(f, last_fwd, last_fwd_change)
    bwd_edge, last_bwd, last_bwd_change = pressed_edge(b, last_bwd, last_bwd_change)
    stop_edge, last_stop, last_stop_change = pressed_edge(s, last_stop, last_stop_change)


    if state != 0 and duty == 0:
        state = 0

    
    if stop_edge == 1:
        state = 0

    
    if fwd_edge == 1:
        if duty > 0 and state == 0:
            state = 1

    
    if bwd_edge == 1:
        if duty > 0 and state == 0:
            state = 2

    
    if state == 0:
        stby.value(0)
        in1.value(0)
        in2.value(0)
        pwm.duty_u16(0)

    elif state == 1:
        stby.value(1)
        in1.value(1)
        in2.value(0)
        pwm.duty_u16(duty)

    elif state == 2:
        stby.value(1)
        in1.value(0)
        in2.value(1)
        pwm.duty_u16(duty)

  
    now = time.ticks_ms()
    if time.ticks_diff(now, last_print_ms) >= PRINT_EVERY_MS:
        print("Estado:", state_text(state),
              "| FWD:", f, "BWD:", b, "STOP:", s,
              "| RAW:", raw, "| V:", "{:.2f}".format(volts),
              "| DUTY:", duty,
              "| edges:", fwd_edge, bwd_edge, stop_edge)
        last_print_ms = now

    time.sleep_ms(10)
