#include <stdio.h>
#include "freertos/FreeRTOS.h"
#include "freertos/task.h"
#include "driver/adc.h"
#include "driver/gpio.h"
#include "driver/ledc.h"

// ===== JOYSTICK 1 =====
#define JOY1_X ADC1_CHANNEL_6   // GPIO34
#define JOY1_Y ADC1_CHANNEL_7   // GPIO35

// ===== JOYSTICK 2 =====
#define JOY2_X ADC1_CHANNEL_5   // GPIO33

// ===== SERVOS =====
#define SERVO_RUDDER   14
#define SERVO_ELEV_L   12
#define SERVO_ELEV_R   27
#define SERVO_AIL_L    23
#define SERVO_AIL_R    16

// ===== PWM =====
#define SERVO_FREQ 50
#define SERVO_RES LEDC_TIMER_15_BIT

#define CENTER 1500
#define RANGE  400
#define DEAD   25

// ---------- helpers ----------

uint32_t us_to_duty(int us){
    uint32_t max = (1 << 15) - 1;
    return (us * max) / 20000;
}

void servo_write(int ch, int us){
    ledc_set_duty(LEDC_LOW_SPEED_MODE, ch, us_to_duty(us));
    ledc_update_duty(LEDC_LOW_SPEED_MODE, ch);
}

int adc_avg(adc1_channel_t ch){
    int s=0;
    for(int i=0;i<6;i++) s += adc1_get_raw(ch);
    return s/6;
}

int map_axis(int v){
    int d = v - 2048;
    if (d > -DEAD && d < DEAD) return 0;
    return (d * RANGE) / 2048;
}

// ---------- main ----------

void app_main(void)
{
    // ADC config
    adc1_config_width(ADC_WIDTH_BIT_12);
    adc1_config_channel_atten(JOY1_X, ADC_ATTEN_DB_11);
    adc1_config_channel_atten(JOY1_Y, ADC_ATTEN_DB_11);
    adc1_config_channel_atten(JOY2_X, ADC_ATTEN_DB_11);

    // PWM timer
    ledc_timer_config_t timer = {
        .speed_mode = LEDC_LOW_SPEED_MODE,
        .timer_num = LEDC_TIMER_0,
        .duty_resolution = SERVO_RES,
        .freq_hz = SERVO_FREQ
    };
    ledc_timer_config(&timer);

    int pins[5] = {
        SERVO_RUDDER,
        SERVO_ELEV_L,
        SERVO_ELEV_R,
        SERVO_AIL_L,
        SERVO_AIL_R
    };

    for(int i=0;i<5;i++){
        ledc_channel_config_t ch={
            .gpio_num = pins[i],
            .speed_mode = LEDC_LOW_SPEED_MODE,
            .channel = i,
            .timer_sel = LEDC_TIMER_0,
            .duty = 0
        };
        ledc_channel_config(&ch);
    }

    // ===== LOOP =====
    while(1){

        int x1 = adc_avg(JOY1_X);
        int y1 = adc_avg(JOY1_Y);
        int x2 = adc_avg(JOY2_X);

        int rudder = map_axis(x1);
        int elev   = map_axis(y1);
        int ail    = map_axis(x2);

        // ---- Timón ----
        servo_write(0, CENTER + rudder);

        // ---- Elevadores ----
        servo_write(1, CENTER + elev);
        servo_write(2, CENTER - elev);

        // ---- Alerones ----
        servo_write(3, CENTER + ail);
        servo_write(4, CENTER + ail);

        vTaskDelay(pdMS_TO_TICKS(70));
    }
}
