#include <stdio.h>
#include <stdlib.h>
#include <string>
#include "pico/stdlib.h"
#include "hardware/i2c.h"
#include "hardware/pwm.h"
#include "pico/stdio_usb.h"

using namespace std;

/* ------------- PINS ------------ */
#define I2C_PORT i2c0
#define DS3231_ADDR 0x68
#define SDA_PIN 4      /* Pins to connect the DS3231 we must make sure that if they are SDA and SCL */
#define SCL_PIN 5

#define MOTOR_PWM_PIN 19 /* Pin to connect the motor PWM */
#define MOTOR_AIN1    18 /* Pins to control motor direction */
#define MOTOR_AIN2    17
#define BUZZER_PIN    13 /* Pin to connect the buzzer */
#define BOTON_PIN     16 /* Pin to connect the button */

/* ------------- CONTROL VARIABLES ------------ */
int h_alarma = -1;
int m_alarma = -1;
string buffer_entrada = "";
int estado_config = 0;

uint32_t ultimo_reloj_ms = 0;
uint32_t ultimo_movimiento_ms = 0;
int fase_alarma = 0; 

bool alarma_disparada = false;
bool ya_se_apago_este_minuto = false;

// With this part of the code we convert from binary to decimal.

/* The watch stores the data in two parts that are a binary number and a binary number. 
For example, 25 looks like this: 0010 (2) 0101 (5) Then it becomes val >> 4: Move everything to the right to keep only the 2.
Then multiply it by 10 (20). 
And val & 0x0F: Delete the left side to keep only the 5. then add it and there are 25. */
uint8_t bcd_to_dec(uint8_t val) { 
    return ((val >> 4) * 10) + (val & 0x0F); 
}

/* ------------ MOTOR ----------------*/
void setup_motor() {
    gpio_set_function(MOTOR_PWM_PIN, GPIO_FUNC_PWM); /* configure the pin for the motor with PWM */
    pwm_set_wrap(pwm_gpio_to_slice_num(MOTOR_PWM_PIN), 255);
    pwm_set_enabled(pwm_gpio_to_slice_num(MOTOR_PWM_PIN), true);
    
    // We initialize our pins as output
    gpio_init(MOTOR_AIN1); gpio_set_dir(MOTOR_AIN1, GPIO_OUT);
    gpio_init(MOTOR_AIN2); gpio_set_dir(MOTOR_AIN2, GPIO_OUT);
}

void mover_motor(int speed) {
    if (speed > 0) { 
        gpio_put(MOTOR_AIN1, 1); gpio_put(MOTOR_AIN2, 0); 
        pwm_set_gpio_level(MOTOR_PWM_PIN, speed); 
    }
    else if (speed < 0) { 
        gpio_put(MOTOR_AIN1, 0); gpio_put(MOTOR_AIN2, 1); 
        pwm_set_gpio_level(MOTOR_PWM_PIN, -speed); 
    }
    else { 
        gpio_put(MOTOR_AIN1, 0); gpio_put(MOTOR_AIN2, 0); 
        pwm_set_gpio_level(MOTOR_PWM_PIN, 0); 
    }
}

/* ------------ BUZZER ------------- */
void setup_buzzer() {
    gpio_set_function(BUZZER_PIN, GPIO_FUNC_PWM);
    uint slice = pwm_gpio_to_slice_num(BUZZER_PIN);
    pwm_set_clkdiv(slice, 4.0f); /* Reduce base frequency */
    pwm_set_enabled(slice, true);
}

void sonar_buzzer(uint freq) { /* give it specific sound commands */
    uint slice = pwm_gpio_to_slice_num(BUZZER_PIN);
    if (freq == 0) {
        pwm_set_gpio_level(BUZZER_PIN, 0); /* Line to prevent our buzzer from sounding */
        return;
    }
    uint32_t wrap = 125000000 / 4 / freq;
    pwm_set_wrap(slice, wrap);
    pwm_set_gpio_level(BUZZER_PIN, wrap / 2); /* This line calculates the vibrations (freq) you want to hear */
}

/* ------------ KEYBOARD CONFIGURATION ------------- */
void procesar_teclado() { /* For the user to enter the time */
    int ch = getchar_timeout_us(0);
    if (ch == PICO_ERROR_TIMEOUT) return;
    char c = (char)ch;
    
    if (c == ';') {
        if (buffer_entrada.length() > 0) {
            int valor = stoi(buffer_entrada);
            if (estado_config == 0) {
                h_alarma = valor;
                printf("\n[OK] Hour saved. Now enter minutes (ex: 30;): ");
                estado_config = 1;
            }
            else if (estado_config == 1) {
                m_alarma = valor;
                printf("\n[OK] Minutes saved. SYSTEM ACTIVATED.\n");
                estado_config = 2;
            }
            buffer_entrada = "";
        }
    } else if (c >= '0' && c <= '9') {
        buffer_entrada += c;
        putchar(c);
    }
}

/* --------------- MAIN ------------------- */
int main() {
    stdio_init_all();

    /* Wait for the USB terminal to be ready */
    while (!stdio_usb_connected()) { sleep_ms(100); }
    sleep_ms(1000);

    /* Clear screen and start configuration */
    printf("\033[2J\033[H"); 
    printf("Hamster Alarm Configuration\n");
    printf("Enter the alarm HOUR (ex: 7;): ");

    i2c_init(I2C_PORT, 100000);
    gpio_set_function(SDA_PIN, GPIO_FUNC_I2C);
    gpio_set_function(SCL_PIN, GPIO_FUNC_I2C);
    gpio_pull_up(SDA_PIN); gpio_pull_up(SCL_PIN);
   
    setup_motor();
    setup_buzzer();
    
    gpio_init(BOTON_PIN);
    gpio_set_dir(BOTON_PIN, GPIO_IN);
    gpio_pull_up(BOTON_PIN);

    uint8_t reg = 0x00;
    uint8_t data[3];

    while (true) {
        uint32_t ahora_ms = to_ms_since_boot(get_absolute_time());

        if (estado_config < 2) {
            procesar_teclado();
        }

        /* Read the RTC (DS3231) every 500ms */
        static uint32_t ultima_lectura_rtc = 0;
        if (ahora_ms - ultima_lectura_rtc > 500) {
            i2c_write_blocking(I2C_PORT, DS3231_ADDR, &reg, 1, true);
            i2c_read_blocking(I2C_PORT, DS3231_ADDR, data, 3, false);
            ultima_lectura_rtc = ahora_ms;
        }
       
        uint8_t min = bcd_to_dec(data[1]);
        uint8_t hor = bcd_to_dec(data[2] & 0x3F);

        /* activate the alarm at the time the user logged in */
        if (estado_config == 2) {
            if (ahora_ms - ultimo_reloj_ms >= 5000) {
                printf("Current Time: %02d:%02d | Alarm: %02d:%02d\n", hor, min, h_alarma, m_alarma);
                ultimo_reloj_ms = ahora_ms;
            }
           
            if (hor == h_alarma && min == m_alarma) {
                if (!alarma_disparada && !ya_se_apago_este_minuto) {
                    alarma_disparada = true;
                    printf("\n¡Alarma activada!\n");
                }
            } else {
                ya_se_apago_este_minuto = false;
            }
        }

        /* if the alarm is activate the buzzer and the motor are activated */
        if (alarma_disparada) {
            /* if the button is activated the buzzer and the motor are desactivated */
            if (gpio_get(BOTON_PIN) == 0) {
                alarma_disparada = false;
                ya_se_apago_este_minuto = true; 
                mover_motor(0);
                sonar_buzzer(0);
                printf("\nAlarm turned off.\n");
                sleep_ms(1000);
            } else {
                /* Movement cycle: 2 seconds forward and 2 seconds backward */
                if (ahora_ms - ultimo_movimiento_ms >= 2000) {
                    if (fase_alarma == 0) {
                        mover_motor(230); // Move forward
                        sonar_buzzer(1800);
                        fase_alarma = 1;
                    } else {
                        mover_motor(-230); // Move backward
                        sonar_buzzer(1300);
                        fase_alarma = 0;
                    }
                    ultimo_movimiento_ms = ahora_ms;
                }
            }
        }
        sleep_ms(1);
    }
}