/* FabAcademy starter JS (no frameworks) */

function getBasePath(){
  // Determine depth from current path (supports /assignments/ pages)
  const path = window.location.pathname.replace(/\/+$/, '');
  const parts = path.split('/').filter(Boolean);

  // If last part has a dot, it's a file; otherwise it's a folder index
  const isFile = parts.length && parts[parts.length - 1].includes('.');
  const depth = Math.max(0, (isFile ? parts.length - 1 : parts.length) - 1);
  return depth === 0 ? './' : '../'.repeat(depth);
}

async function loadPartial(targetId, file){
  const base = getBasePath();
  const el = document.getElementById(targetId);
  if(!el) return;

  try{
    const res = await fetch(base + file);
    const html = await res.text();
    el.innerHTML = html;

    // Fix nav/footer links that use data-nav
    el.querySelectorAll('a[data-nav]').forEach(a => {
      const href = a.getAttribute('data-nav');
      a.setAttribute('href', base + href);
    });

    // Fix image src in partials
    el.querySelectorAll('[data-src]').forEach(node => {
      node.setAttribute('src', base + node.getAttribute('data-src'));
    });
  }catch(err){
    console.error('Error loading ' + file + ':', err);
  }
}

/* Back to top */
function setupBackToTop(){
  const btn = document.querySelector('.back-to-top');
  if(!btn) return;

  const toggle = () => {
    if(window.scrollY > 600) btn.classList.add('is-visible');
    else btn.classList.remove('is-visible');
  };

  window.addEventListener('scroll', toggle, {passive:true});
  toggle();

  btn.addEventListener('click', () => {
    window.scrollTo({top:0, behavior:'smooth'});
  });
}

/* Terminal copy buttons (copies <pre> inside same component) */
async function copyTextToClipboard(text){
  try{
    await navigator.clipboard.writeText(text);
    return true;
  }catch(err){
    console.error('Clipboard error:', err);
    return false;
  }
}

function setupTerminalCopy(){
  document.addEventListener('click', async (e) => {
    const btn = e.target.closest('[data-copy-terminal]');
    if(!btn) return;

    const terminal = btn.closest('.terminal');
    if(!terminal) return;

    const pre = terminal.querySelector('pre');
    if(!pre) return;

    const ok = await copyTextToClipboard(pre.textContent.replace(/\n$/, ''));
    const old = btn.textContent;
    btn.textContent = ok ? 'Copied ✓' : 'Copy failed';
    setTimeout(() => { btn.textContent = old; }, 1200);
  });
}

/* Lightbox: fill captions from <img alt=""> */
function fillLightboxCaptions(){
  document.querySelectorAll('.lightbox').forEach(lb => {
    const img = lb.querySelector('img');
    const cap = lb.querySelector('.lb-caption');
    if(img && cap){
      cap.textContent = img.alt || '';
    }
  });
}

/* ===== Collapsible code blocks: auto line numbers + copy + bottom bar ===== */

function convertCodeBlocks(){
  document.querySelectorAll('.codeblock .code-raw code').forEach(codeEl => {
    const pre = codeEl.closest('.code-raw');
    if(!pre) return;

    let text = codeEl.textContent;

    if(text.startsWith('\n')) text = text.slice(1);
    text = text.replace(/\n$/, '');

    const lines = text.split('\n');

    const wrapper = document.createElement('div');
    wrapper.className = 'code-numbered';

    lines.forEach(line => {
      const lineDiv = document.createElement('div');
      lineDiv.className = 'code-line';
      lineDiv.textContent = line;
      wrapper.appendChild(lineDiv);
    });

    pre.replaceWith(wrapper);
  });
}

function addCodeFooters(){
  document.querySelectorAll('.codeblock').forEach(details => {
    if(details.querySelector('.code-footer')) return;

    const titleEl = details.querySelector('.code-title');
    const titleText = titleEl ? titleEl.textContent : 'code';

    const footer = document.createElement('div');
    footer.className = 'code-footer';
    footer.innerHTML = `
      <span class="code-title"></span>
      <span class="code-actions">
        <span class="code-toggle" data-toggle-code></span>
        <button class="copy-btn" type="button" data-copy-code>Copy</button>
      </span>
    `;

    footer.querySelector('.code-title').textContent = titleText;
    details.appendChild(footer);
  });
}

function setToggleText(el, text){
  if(!el) return;
  el.textContent = text;
}

function syncCodeToggleLabels(details){
  const summaryToggle = details.querySelector('summary .code-toggle');
  setToggleText(summaryToggle, details.open ? 'Hide code' : 'Show code');

  const footerToggle = details.querySelector('.code-footer [data-toggle-code]');
  setToggleText(footerToggle, 'Hide code');
}

async function copyCodeFromBlock(btn){
  const block = btn.closest('.codeblock');
  if(!block) return;

  const numbered = block.querySelector('.code-numbered');
  if(!numbered) return;

  const lines = Array.from(numbered.querySelectorAll('.code-line')).map(el => el.textContent);
  const text = lines.join('\n');

  const ok = await copyTextToClipboard(text);
  const old = btn.textContent;
  btn.textContent = ok ? 'Copied ✓' : 'Copy failed';
  setTimeout(() => { btn.textContent = old; }, 1200);
}

function setupCodeBlocks(){
  convertCodeBlocks();
  addCodeFooters();

  document.querySelectorAll('.codeblock').forEach(details => {
    syncCodeToggleLabels(details);
    details.addEventListener('toggle', () => syncCodeToggleLabels(details));
  });

  document.addEventListener('click', (e) => {
    const copyBtn = e.target.closest('[data-copy-code]');
    if(copyBtn){
      copyCodeFromBlock(copyBtn);
      return;
    }

    const toggleBtn = e.target.closest('[data-toggle-code]');
    if(toggleBtn){
      const details = toggleBtn.closest('.codeblock');
      if(details) details.open = false;
    }
  });
}

document.addEventListener('DOMContentLoaded', async () => {
  await loadPartial('nav', 'nav.html');
  await loadPartial('footer', 'footer.html');

  setupBackToTop();
  setupTerminalCopy();
  fillLightboxCaptions();
  setupCodeBlocks();
});

// IMAGE and VIDEO size on figcaption
async function addMediaSizesToFigures() {
  // Select all <figure> elements that contain a figcaption and either an <img> or a <video>
  const figures = Array.from(document.querySelectorAll("figure")).filter(fig => {
    return fig.querySelector("figcaption") && (fig.querySelector("img") || fig.querySelector("video"));
  });

  // Cache sizes by URL to avoid repeated requests
  const cache = new Map();

  // Try to get size via HEAD (Content-Length). If missing, use a Range request (Content-Range).
  async function getSizeInBytes(url) {
    if (cache.has(url)) return cache.get(url);

    // 1) Try HEAD
    try {
      const headRes = await fetch(url, { method: "HEAD" });
      const len = headRes.headers.get("content-length");
      if (len) {
        const bytes = Number(len);
        cache.set(url, bytes);
        return bytes;
      }
    } catch (_) {
      // Ignore and fallback
    }

    // 2) Fallback: Range request (downloads only 1 byte in most servers)
    try {
      const rangeRes = await fetch(url, {
        method: "GET",
        headers: { Range: "bytes=0-0" }
      });

      const contentRange = rangeRes.headers.get("content-range"); // e.g. "bytes 0-0/123456"
      if (contentRange) {
        const match = contentRange.match(/\/(\d+)\s*$/);
        if (match) {
          const bytes = Number(match[1]);
          cache.set(url, bytes);
          return bytes;
        }
      }

      // Some servers may still provide Content-Length for the 1-byte response,
      // but that's NOT the full size, so we don't use it here.
    } catch (_) {
      // Ignore
    }

    cache.set(url, NaN);
    return NaN;
  }

  function formatSize(bytes) {
    // Format size as Kb or Mb (using 1024 base)
    if (!Number.isFinite(bytes)) return null;
    if (bytes >= 1024 * 1024) return (bytes / (1024 * 1024)).toFixed(1) + "Mb";
    return Math.round(bytes / 1024) + "Kb";
  }

  for (const fig of figures) {
    const caption = fig.querySelector("figcaption");
    const img = fig.querySelector("img");
    const video = fig.querySelector("video");

    const isVideo = !!video && !img;
    const mediaEl = img || video;

    if (!mediaEl || !caption) continue;

    // Detect lightbox only for images (img inside <a>)
    const hasLightbox = !isVideo && !!mediaEl.closest("a");

    // Resolve URL
    let url = "";
    if (isVideo) {
      const source = video.querySelector("source");
      url = (source && source.src) ? source.src : (video.src || "");
    } else {
      url = img.currentSrc || img.src || "";
    }

    if (!url) continue;

    const bytes = await getSizeInBytes(url);
    const sizeText = formatSize(bytes);
    if (!sizeText) continue;

    // Remove any previous size fragment from the caption
    let baseText = caption.textContent
      .replace(/\(.*?(size|image size|video size).*?\)/gi, "")
      .replace(/[-–—]?\s*click to enlarge\s*[-–—]?/gi, "")
      .trim();

    // Build suffix depending on media type and lightbox presence
    let suffix = "";
    if (isVideo) {
      suffix = `(video size: ${sizeText}).`;
    } else {
      suffix = hasLightbox
        ? `(size ${sizeText} -click to enlarge-)`
        : `(image size: ${sizeText}).`;
    }

    caption.textContent = `${baseText} ${suffix}`.trim();
  }
}

// Run when DOM is ready
if (document.readyState === "loading") {
  document.addEventListener("DOMContentLoaded", addMediaSizesToFigures);
} else {
  addMediaSizesToFigures();
}

