// ===== Collapsible code blocks: auto line numbers + copy + bottom bar =====

// Convert <pre class="code-raw"><code>...</code></pre> into numbered lines
function convertCodeBlocks(){
  document.querySelectorAll('.codeblock .code-raw code').forEach(codeEl => {
    const pre = codeEl.closest('.code-raw');
    if(!pre) return;

    let text = codeEl.textContent;

    if(text.startsWith('\n')) text = text.slice(1);
    text = text.replace(/\n$/, '');

    const lines = text.split('\n');

    const wrapper = document.createElement('div');
    wrapper.className = 'code-numbered';

    lines.forEach(line => {
      const lineDiv = document.createElement('div');
      lineDiv.className = 'code-line';
      lineDiv.textContent = line;
      wrapper.appendChild(lineDiv);
    });

    pre.replaceWith(wrapper);
  });
}

// Insert bottom bar (title + toggle + copy)
function addCodeFooters(){
  document.querySelectorAll('.codeblock').forEach(details => {
    if(details.querySelector('.code-footer')) return;

    const titleEl = details.querySelector('.code-title');
    const titleText = titleEl ? titleEl.textContent : 'code';

    const footer = document.createElement('div');
    footer.className = 'code-footer';
    footer.innerHTML = `
      <span class="code-title"></span>
      <span class="code-actions">
        <span class="code-toggle" data-toggle-code></span>
        <button class="copy-btn" type="button" data-copy-code>Copy</button>
      </span>
    `;

    footer.querySelector('.code-title').textContent = titleText;

    details.appendChild(footer);
  });
}

// Ensure toggles contain ONLY text (no triangles), icons come from CSS ::before
function setToggleText(el, text){
  if(!el) return;
  el.textContent = text; // wipes any previous "◀" etc.
}

function syncCodeToggleLabels(details){
  const summaryToggle = details.querySelector('summary .code-toggle');
  setToggleText(summaryToggle, details.open ? 'Hide code' : 'Show code');

  const footerToggle = details.querySelector('.code-footer [data-toggle-code]');
  setToggleText(footerToggle, 'Hide code');
}

// Copy: copy code text (without numbers)
async function copyCodeFromBlock(btn){
  const block = btn.closest('.codeblock');
  if(!block) return;

  const numbered = block.querySelector('.code-numbered');
  if(!numbered) return;

  const lines = Array.from(numbered.querySelectorAll('.code-line')).map(el => el.textContent);
  const text = lines.join('\n');

  try{
    await navigator.clipboard.writeText(text);
    const old = btn.textContent;
    btn.textContent = 'Copied ✓';
    setTimeout(() => { btn.textContent = old; }, 1200);
  }catch(err){
    console.error('Clipboard error:', err);
  }
}

window.addEventListener('DOMContentLoaded', () => {
  convertCodeBlocks();
  addCodeFooters();

  document.querySelectorAll('.codeblock').forEach(details => {
    syncCodeToggleLabels(details);
    details.addEventListener('toggle', () => syncCodeToggleLabels(details));
  });

  document.addEventListener('click', (e) => {
    const copyBtn = e.target.closest('[data-copy-code]');
    if(copyBtn){
      copyCodeFromBlock(copyBtn);
      return;
    }

    const toggleBtn = e.target.closest('[data-toggle-code]');
    if(toggleBtn){
      const details = toggleBtn.closest('.codeblock');
      if(details) details.open = false;
    }
  });
});
