#!/usr/bin/env python3
"""
view_serial_stream.py
---------------------
Receives JPEG frames from the XIAO ESP32-S3 Sense camera over USB serial
and displays them in an OpenCV window with a live FPS counter.

Each frame is preceded by a 12-byte FrameHeader:
  [4 bytes] magic: "JPG0"
  [4 bytes] jpeg_len (uint32, little-endian)
  [2 bytes] width   (uint16, little-endian)
  [2 bytes] height  (uint16, little-endian)

Usage:
  pip install pyserial opencv-python numpy
  python view_serial_stream.py
  python view_serial_stream.py --port /dev/cu.usbmodem1101
  Press 'q' or Escape to quit.
"""

import argparse
import struct
import time

import cv2
import numpy as np
import serial
from serial.tools import list_ports

MAGIC = b"JPG0"
HEADER_FMT  = "<4sIHH"   # magic(4) + jpeg_len(4) + width(2) + height(2)
HEADER_SIZE = struct.calcsize(HEADER_FMT)


def choose_port(explicit):
    if explicit:
        return explicit
    for p in list(list_ports.comports()):
        if "usbmodem" in p.device or "ttyACM" in p.device or "ttyUSB" in p.device:
            return p.device
    raise RuntimeError("No serial camera port found. Pass --port.")


def read_exact(ser, n):
    """Block until exactly n bytes have been received."""
    out = bytearray()
    while len(out) < n:
        chunk = ser.read(n - len(out))
        if not chunk:
            raise TimeoutError("Timed out waiting for serial data")
        out.extend(chunk)
    return bytes(out)


def sync_to_magic(ser):
    """Scan incoming bytes until the 4-byte magic sequence is found, then
    read the rest of the header and return the complete header bytes."""
    window = bytearray()
    while True:
        b = ser.read(1)
        if not b:
            raise TimeoutError("Timed out while syncing to frame boundary")
        window += b
        if len(window) > 4:
            window.pop(0)
        if bytes(window) == MAGIC:
            rest = read_exact(ser, HEADER_SIZE - 4)
            return MAGIC + rest


def main():
    parser = argparse.ArgumentParser(description="View ESP32 serial JPEG stream")
    parser.add_argument("--port",    default=None, help="Serial port")
    parser.add_argument("--baud",    type=int,   default=3000000)
    parser.add_argument("--timeout", type=float, default=2.0)
    args = parser.parse_args()

    port = choose_port(args.port)
    print(f"Opening {port} @ {args.baud} baud")

    ser = serial.Serial(port=port, baudrate=args.baud, timeout=args.timeout)
    time.sleep(1.0)
    ser.reset_input_buffer()
    ser.write(b"S")   # tell the ESP32 to start streaming

    frames = 0
    fps_t0 = time.time()
    fps    = 0.0

    try:
        while True:
            header_bytes = sync_to_magic(ser)
            magic, jpeg_len, width, height = struct.unpack(HEADER_FMT, header_bytes)

            # Sanity-check the header before allocating memory
            if magic != MAGIC or jpeg_len == 0 or jpeg_len > 700_000:
                continue

            jpeg  = read_exact(ser, jpeg_len)
            arr   = np.frombuffer(jpeg, dtype=np.uint8)
            frame = cv2.imdecode(arr, cv2.IMREAD_COLOR)
            if frame is None:
                continue

            frames += 1
            dt = time.time() - fps_t0
            if dt >= 1.0:
                fps    = frames / dt
                frames = 0
                fps_t0 = time.time()

            cv2.putText(frame, f"{width}x{height}  {fps:.1f} fps",
                        (10, 24), cv2.FONT_HERSHEY_SIMPLEX, 0.65,
                        (0, 255, 0), 2, cv2.LINE_AA)
            cv2.imshow("ESP32 Serial Camera", frame)

            if cv2.waitKey(1) & 0xFF in (ord("q"), 27):
                break

    except KeyboardInterrupt:
        pass
    finally:
        try:
            ser.write(b"X")   # tell the ESP32 to stop
            time.sleep(0.1)
        except Exception:
            pass
        ser.close()
        cv2.destroyAllWindows()


if __name__ == "__main__":
    main()
