/*
 * XIAO ESP32-S3 Sense — Camera to USB Serial Stream
 * ===================================================
 * Captures JPEG frames from the OV2640 camera and sends them over USB
 * serial at 3 Mbaud to a host computer.  A companion Python script
 * (tools/view_serial_stream.py) receives and displays the frames.
 *
 * Commands (single byte, sent from host):
 *   'S' — start continuous streaming
 *   'X' — stop streaming
 *   'C' — capture and send one frame immediately
 *
 * Frame format over serial:
 *   [FrameHeader: 12 bytes][JPEG data: header.jpegLen bytes]
 *
 * The FrameHeader starts with the magic bytes "JPG0" so the receiver
 * can re-synchronise if bytes are lost.
 *
 * Hardware:
 *   XIAO ESP32-S3 Sense with the OV2640 camera board attached via the
 *   on-board board-to-board connector (no extra wiring needed).
 */

#include <Arduino.h>
#include <cstring>
#include "esp_camera.h"

namespace {

// OV2640 camera pin map for the XIAO ESP32-S3 Sense.
// These are fixed by the PCB design — do not change.
constexpr int PWDN_GPIO_NUM  = -1;
constexpr int RESET_GPIO_NUM = -1;
constexpr int XCLK_GPIO_NUM  = 10;
constexpr int SIOD_GPIO_NUM  = 40;   // SCCB (I2C-like) SDA
constexpr int SIOC_GPIO_NUM  = 39;   // SCCB SCL

// 8-bit parallel pixel data bus (Y2..Y9)
constexpr int Y9_GPIO_NUM  = 48;
constexpr int Y8_GPIO_NUM  = 11;
constexpr int Y7_GPIO_NUM  = 12;
constexpr int Y6_GPIO_NUM  = 14;
constexpr int Y5_GPIO_NUM  = 16;
constexpr int Y4_GPIO_NUM  = 18;
constexpr int Y3_GPIO_NUM  = 17;
constexpr int Y2_GPIO_NUM  = 15;

// Synchronisation signals
constexpr int VSYNC_GPIO_NUM = 38;
constexpr int HREF_GPIO_NUM  = 47;
constexpr int PCLK_GPIO_NUM  = 13;

// Streaming configuration
constexpr uint32_t    SERIAL_BAUD       = 3000000;
constexpr uint32_t    FRAME_INTERVAL_MS = 170;          // ~6 fps
constexpr framesize_t STREAM_FRAME_SIZE = FRAMESIZE_VGA; // 640x480
constexpr int         STREAM_JPEG_QUALITY = 12;          // 0=best, 63=worst

// A fixed-size header sent before every JPEG frame.
// The magic bytes allow the receiver to find frame boundaries even if
// some bytes are corrupted or the receiver starts mid-stream.
struct FrameHeader {
    char     magic[4];    // always "JPG0"
    uint32_t jpegLen;     // number of JPEG bytes that follow
    uint16_t width;       // frame width in pixels
    uint16_t height;      // frame height in pixels
};

bool     g_streaming   = false;
uint32_t g_lastFrameMs = 0;

// Initialise the OV2640 and configure it for JPEG output.
bool initCamera() {
    camera_config_t config;
    config.ledc_channel = LEDC_CHANNEL_0;
    config.ledc_timer   = LEDC_TIMER_0;
    config.pin_d0       = Y2_GPIO_NUM;
    config.pin_d1       = Y3_GPIO_NUM;
    config.pin_d2       = Y4_GPIO_NUM;
    config.pin_d3       = Y5_GPIO_NUM;
    config.pin_d4       = Y6_GPIO_NUM;
    config.pin_d5       = Y7_GPIO_NUM;
    config.pin_d6       = Y8_GPIO_NUM;
    config.pin_d7       = Y9_GPIO_NUM;
    config.pin_xclk     = XCLK_GPIO_NUM;
    config.pin_pclk     = PCLK_GPIO_NUM;
    config.pin_vsync    = VSYNC_GPIO_NUM;
    config.pin_href     = HREF_GPIO_NUM;
    config.pin_sccb_sda = SIOD_GPIO_NUM;
    config.pin_sccb_scl = SIOC_GPIO_NUM;
    config.pin_pwdn     = PWDN_GPIO_NUM;
    config.pin_reset    = RESET_GPIO_NUM;
    config.xclk_freq_hz = 20000000;    // 20 MHz pixel clock
    config.pixel_format = PIXFORMAT_JPEG;

    config.frame_size   = STREAM_FRAME_SIZE;
    config.jpeg_quality = STREAM_JPEG_QUALITY;
    config.fb_count     = 1;
    config.grab_mode    = CAMERA_GRAB_WHEN_EMPTY;

    const esp_err_t err = esp_camera_init(&config);
    if (err != ESP_OK) {
        Serial.printf("Camera init failed: 0x%x\n", err);
        return false;
    }

    sensor_t* sensor = esp_camera_sensor_get();
    if (sensor) {
        sensor->set_vflip(sensor, 0);
        sensor->set_hmirror(sensor, 0);
        sensor->set_framesize(sensor, STREAM_FRAME_SIZE);
        sensor->set_quality(sensor, STREAM_JPEG_QUALITY);
    }

    return true;
}

// Capture one frame and write it to serial.
void sendFrame() {
    camera_fb_t* fb = esp_camera_fb_get();
    if (!fb) return;

    if (fb->format != PIXFORMAT_JPEG || !fb->buf || fb->len == 0) {
        esp_camera_fb_return(fb);
        return;
    }

    // Send the header so the receiver knows how many bytes to read.
    FrameHeader header;
    memcpy(header.magic, "JPG0", 4);
    header.jpegLen = fb->len;
    header.width   = static_cast<uint16_t>(fb->width);
    header.height  = static_cast<uint16_t>(fb->height);

    Serial.write(reinterpret_cast<const uint8_t*>(&header), sizeof(header));
    Serial.write(fb->buf, fb->len);

    esp_camera_fb_return(fb);
}

// Handle single-byte commands from the host.
void processCommands() {
    while (Serial.available() > 0) {
        const int c = Serial.read();
        if      (c == 'S') { g_streaming = true;  g_lastFrameMs = 0; }
        else if (c == 'X') { g_streaming = false; }
        else if (c == 'C') { sendFrame(); }
    }
}

}  // namespace

void setup() {
    Serial.begin(SERIAL_BAUD);
    delay(400);

    if (!initCamera()) {
        while (true) delay(1000);  // halt on camera failure
    }

    Serial.println("USB camera serial stream ready");
    Serial.println("Commands: S=start, X=stop, C=capture");
}

void loop() {
    processCommands();

    if (g_streaming) {
        const uint32_t now = millis();
        if (now - g_lastFrameMs >= FRAME_INTERVAL_MS) {
            g_lastFrameMs = now;
            sendFrame();
        }
    }

    delay(1);
}
