/*
 * Ramadan Kareem Animation — XIAO ESP32-S3 + SSD1306 OLED
 * =========================================================
 * A button on D2 triggers a 7-second animation:
 *   - A crescent moon slides from the left side to the centre of the screen.
 *   - Stars twinkle throughout.
 *   - "RAMADAN KAREEM" types itself letter by letter.
 * At the end the final card stays on screen.  Pressing the button again
 * returns to the idle crescent screen.
 *
 * Hardware connections (Grove Shield on XIAO ESP32-S3):
 *   OLED SDA -> D4 (Grove I2C)
 *   OLED SCL -> D5 (Grove I2C)
 *   Button   -> D2
 */

#include <Arduino.h>
#include <Wire.h>
#include <Adafruit_GFX.h>
#include <Adafruit_SSD1306.h>

namespace {

// Pin and display configuration
constexpr int     I2C_SDA        = D4;
constexpr int     I2C_SCL        = D5;
constexpr int     BUTTON_PIN     = D2;
constexpr bool    BUTTON_ACTIVE_LOW = true;
constexpr uint32_t DEBOUNCE_MS   = 25;

constexpr int     SCREEN_W       = 128;
constexpr int     SCREEN_H       = 64;
constexpr int     OLED_RESET     = -1;
constexpr uint8_t OLED_ADDR      = 0x3C;

// Animation timing
constexpr uint32_t ANIM_TEXT_START_MS = 2200;  // text appears 2.2 s in
constexpr uint32_t ANIM_DURATION_MS   = 7000;  // total animation length
constexpr uint8_t  TEXT_SIZE          = 1;
constexpr int16_t  TEXT_X             = 56;
constexpr int16_t  TEXT_Y1            = 26;
constexpr int16_t  TEXT_Y2            = 40;

Adafruit_SSD1306 display(SCREEN_W, SCREEN_H, &Wire, OLED_RESET);

// Button state
bool     gBtnStable       = HIGH;
bool     gBtnLastRead     = HIGH;
uint32_t gBtnLastChangeMs = 0;

// Scene state machine
enum class SceneState : uint8_t { Idle, Animating, Finished };
SceneState gScene      = SceneState::Idle;
uint32_t   gAnimStartMs = 0;
uint32_t   gFrameTick   = 0;

// Fixed star positions (chosen to look natural and not overlap the text area)
struct Star { int16_t x, y; };
constexpr Star STARS[] = {
    {9, 12}, {22, 6}, {52, 10}, {85, 8}, {110, 14}, {102, 30}, {16, 44}, {58, 26}
};

// --- Button helpers -----------------------------------------------------------

bool isPressed(bool pinLevel) {
    return BUTTON_ACTIVE_LOW ? !pinLevel : pinLevel;
}

bool pollButtonPressed() {
    const bool    current = static_cast<bool>(digitalRead(BUTTON_PIN));
    const uint32_t now    = millis();

    if (current != gBtnLastRead) {
        gBtnLastRead     = current;
        gBtnLastChangeMs = now;
    }

    if ((now - gBtnLastChangeMs) > DEBOUNCE_MS && gBtnStable != gBtnLastRead) {
        gBtnStable = gBtnLastRead;
        if (isPressed(gBtnStable)) return true;
    }
    return false;
}

// --- Drawing functions -------------------------------------------------------

// Crescent shape: draw a filled circle, then cover the right portion with a
// slightly offset black circle to carve out the crescent.
void drawCrescent(int16_t cx, int16_t cy, int16_t radius) {
    display.fillCircle(cx, cy, radius, SSD1306_WHITE);
    display.fillCircle(cx + radius / 2, cy - radius / 5, radius, SSD1306_BLACK);
}

// Stars flicker by skipping every third star on alternating frames.
// The modulo values were tuned by eye to give a natural feel.
void drawTwinkleStars(uint32_t tick) {
    for (size_t i = 0; i < (sizeof(STARS) / sizeof(STARS[0])); ++i) {
        if (((tick / 7) + i) % 3 != 0) {
            display.drawPixel(STARS[i].x, STARS[i].y, SSD1306_WHITE);
            // Every so often add cross arms to make the star bigger.
            if (((tick / 10) + i) % 5 == 0) {
                display.drawPixel(STARS[i].x + 1, STARS[i].y, SSD1306_WHITE);
                display.drawPixel(STARS[i].x - 1, STARS[i].y, SSD1306_WHITE);
                display.drawPixel(STARS[i].x, STARS[i].y + 1, SSD1306_WHITE);
                display.drawPixel(STARS[i].x, STARS[i].y - 1, SSD1306_WHITE);
            }
        }
    }
}

// Print only the first visibleChars characters of text, for the typewriter effect.
void drawRevealedWord(int16_t x, int16_t y, const char* text, size_t visibleChars) {
    const size_t len    = strlen(text);
    const size_t toShow = visibleChars > len ? len : visibleChars;
    display.setCursor(x, y);
    for (size_t i = 0; i < toShow; ++i) {
        display.print(text[i]);
    }
}

// --- Scene renderers ---------------------------------------------------------

void startAnimation() {
    gAnimStartMs = millis();
    gFrameTick   = 0;
    gScene       = SceneState::Animating;
}

void drawIdle() {
    display.clearDisplay();
    drawCrescent(SCREEN_W / 2, SCREEN_H / 2, 16);
    display.display();
}

void drawFinalCard() {
    display.clearDisplay();
    drawTwinkleStars(gFrameTick);
    drawCrescent(34, 25, 16);
    display.setTextSize(TEXT_SIZE);
    display.setTextColor(SSD1306_WHITE);
    display.setCursor(TEXT_X, TEXT_Y1); display.print("RAMADAN");
    display.setCursor(TEXT_X, TEXT_Y2); display.print("KAREEM");
    display.display();
}

void drawAnimationFrame(uint32_t elapsed) {
    display.clearDisplay();
    drawTwinkleStars(gFrameTick);

    // Moon slides from off-screen left (-18) to centre (36) over the first 2 s.
    int16_t moonX = (elapsed < 2000)
        ? static_cast<int16_t>(-18 + (54.0f * (static_cast<float>(elapsed) / 2000.0f)))
        : 36;
    drawCrescent(moonX, 25, 16);

    // Text typewriter: starts 2.2 s in and finishes 4.5 s in.
    if (elapsed > ANIM_TEXT_START_MS) {
        const uint32_t revealMs    = elapsed - ANIM_TEXT_START_MS;
        const size_t   totalChars  = strlen("RAMADAN") + strlen("KAREEM");
        const size_t   shown       = static_cast<size_t>(
            min<uint32_t>(totalChars, (revealMs * totalChars) / 2300));

        display.setTextSize(TEXT_SIZE);
        display.setTextColor(SSD1306_WHITE);
        drawRevealedWord(TEXT_X, TEXT_Y1, "RAMADAN", shown);

        size_t secondShown = (shown > strlen("RAMADAN")) ? shown - strlen("RAMADAN") : 0;
        drawRevealedWord(TEXT_X, TEXT_Y2, "KAREEM", secondShown);
    }

    display.display();
}

}  // namespace

// --- Setup & loop ------------------------------------------------------------

void setup() {
    Serial.begin(115200);
    delay(100);

    pinMode(BUTTON_PIN, BUTTON_ACTIVE_LOW ? INPUT_PULLUP : INPUT);
    gBtnStable       = static_cast<bool>(digitalRead(BUTTON_PIN));
    gBtnLastRead     = gBtnStable;
    gBtnLastChangeMs = millis();

    Wire.begin(I2C_SDA, I2C_SCL);
    if (!display.begin(SSD1306_SWITCHCAPVCC, OLED_ADDR)) {
        Serial.println("OLED init failed at 0x3C");
        while (true) delay(100);
    }

    drawIdle();
    Serial.println("Ready. Press D2 to start the animation.");
}

void loop() {
    if (pollButtonPressed()) {
        if (gScene == SceneState::Idle) {
            startAnimation();
        } else {
            // Any press during or after the animation goes back to idle.
            gScene = SceneState::Idle;
            drawIdle();
        }
    }

    const uint32_t now = millis();
    if (gScene == SceneState::Animating) {
        const uint32_t elapsed = now - gAnimStartMs;
        if (elapsed >= ANIM_DURATION_MS) {
            gScene = SceneState::Finished;
            drawFinalCard();
        } else {
            drawAnimationFrame(elapsed);
            ++gFrameTick;
        }
    } else if (gScene == SceneState::Idle) {
        drawIdle();
    } else {
        drawFinalCard();
        ++gFrameTick;
    }

    delay(33);  // ~30 fps
}
