/*
 * RGB LED controlled by a rotary angle sensor
 * =============================================
 * Reads an analog rotary potentiometer and uses the position to set
 * the hue of a Grove Chainable RGB LED.  Rotating the knob cycles
 * through the full colour wheel at constant brightness.
 *
 * The LED speaks the P9813 protocol, which we bit-bang on two GPIO
 * pins (no library needed).
 *
 * Hardware:
 *   Grove Chainable RGB LED  -> D4 (data), D5 (clock)
 *   Grove Rotary Angle Sensor -> A2
 */

#include <Arduino.h>
#include "pins.h"

// --- ADC helpers -------------------------------------------------------------

int readRotaryRawAveraged() {
    // Average 8 readings to reduce ADC noise.
    long sum = 0;
    for (int i = 0; i < 8; ++i) {
        sum += analogRead(PIN_ROTARY);
    }
    return static_cast<int>(sum / 8);
}

// --- P9813 protocol (Grove Chainable RGB LED) --------------------------------
// Each transfer: 32-bit start frame, then one data frame per LED,
// then 32-bit end frame.  Each data frame is:
//   [prefix byte][blue][green][red]
// The prefix encodes inverted top-two bits of each colour channel.

void p9813WriteBit(bool bitVal) {
    digitalWrite(PIN_RGB_DATA, bitVal ? HIGH : LOW);
    digitalWrite(PIN_RGB_CLK, HIGH);
    delayMicroseconds(20);
    digitalWrite(PIN_RGB_CLK, LOW);
    delayMicroseconds(20);
}

void p9813WriteByte(uint8_t b) {
    for (int i = 7; i >= 0; --i) {
        p9813WriteBit((b >> i) & 0x01);
    }
}

uint8_t p9813Prefix(uint8_t r, uint8_t g, uint8_t b) {
    // Start from 0b11000000, then set flag bits for inverted top two bits
    // of B, G, R (in that order) to form the checksum nibble.
    uint8_t p = 0xC0;
    if ((~b) & 0x80) p |= 0x20;
    if ((~b) & 0x40) p |= 0x10;
    if ((~g) & 0x80) p |= 0x08;
    if ((~g) & 0x40) p |= 0x04;
    if ((~r) & 0x80) p |= 0x02;
    if ((~r) & 0x40) p |= 0x01;
    return p;
}

void p9813LatchStart() {
    for (int i = 0; i < 4; ++i) p9813WriteByte(0x00);
}

void p9813LatchEnd() {
    for (int i = 0; i < 4; ++i) p9813WriteByte(0xFF);
}

void setChainableRgb(uint8_t r, uint8_t g, uint8_t b) {
    p9813LatchStart();
    p9813WriteByte(p9813Prefix(r, g, b));
    p9813WriteByte(b);
    p9813WriteByte(g);
    p9813WriteByte(r);
    p9813LatchEnd();
}

// --- Colour conversion -------------------------------------------------------

// Standard integer HSV -> RGB.
// h: 0-255 (hue wraps around the colour wheel)
// s: 0-255 (saturation, 255 = fully saturated)
// v: 0-255 (brightness)
void hsvToRgb(uint8_t h, uint8_t s, uint8_t v,
              uint8_t &r, uint8_t &g, uint8_t &b) {
    if (s == 0) {
        r = g = b = v;
        return;
    }

    uint8_t region    = h / 43;
    uint8_t remainder = (h - (region * 43)) * 6;

    uint8_t p = (v * (255 - s)) >> 8;
    uint8_t q = (v * (255 - ((s * remainder) >> 8))) >> 8;
    uint8_t t = (v * (255 - ((s * (255 - remainder)) >> 8))) >> 8;

    switch (region) {
        case 0:  r = v; g = t; b = p; break;
        case 1:  r = q; g = v; b = p; break;
        case 2:  r = p; g = v; b = t; break;
        case 3:  r = p; g = q; b = v; break;
        case 4:  r = t; g = p; b = v; break;
        default: r = v; g = p; b = q; break;
    }
}

// --- Arduino entry points ----------------------------------------------------

void setup() {
    Serial.begin(115200);
    delay(200);

    pinMode(PIN_RGB_CLK,  OUTPUT);
    pinMode(PIN_RGB_DATA, OUTPUT);
    pinMode(PIN_ROTARY,   INPUT);

    // ESP32-S3 ADC resolution is 12 bits (0..4095).
    analogReadResolution(12);

    digitalWrite(PIN_RGB_CLK,  LOW);
    digitalWrite(PIN_RGB_DATA, LOW);

    // Quick colour test on startup so we know the LED is wired correctly.
    setChainableRgb(255, 0, 0); delay(250);
    setChainableRgb(0, 255, 0); delay(250);
    setChainableRgb(0, 0, 255); delay(250);

    Serial.println("RGB + Rotary ready");
}

void loop() {
    const int raw = readRotaryRawAveraged();

    // Low-pass filter to smooth out ADC jitter while the knob is still.
    static int filtered = raw;
    filtered = (filtered * 7 + raw) / 8;

    // Only commit a new position when the change is large enough to be
    // intentional (12 ADC counts out of 4095 is about 0.3 degrees of rotation).
    static int lastCommitted = filtered;
    if (abs(filtered - lastCommitted) >= 12) {
        lastCommitted = filtered;
    }

    const uint8_t hue = map(lastCommitted, 0, 4095, 0, 255);
    const uint8_t val = 210;  // fixed brightness so only hue changes

    uint8_t r = 0, g = 0, b = 0;
    hsvToRgb(hue, 255, val, r, g, b);

    // Only update the LED hardware when the hue actually changes.
    static uint8_t lastHue = 255;
    if (hue != lastHue) {
        setChainableRgb(r, g, b);
        lastHue = hue;
    }

    // Print a status line every 150 ms for debugging via Serial Monitor.
    static uint32_t lastPrint = 0;
    if (millis() - lastPrint > 150) {
        lastPrint = millis();
        Serial.printf("raw=%d filt=%d committed=%d hue=%u rgb=(%u,%u,%u)\n",
                      raw, filtered, lastCommitted, hue, r, g, b);
    }

    delay(15);
}
