"""
blink_to_stepper.py
--------------------
Reads EEG data from a Muse 2 headband (via muselsl + pylsl) and sends a
"BLINK" command to the Arduino over USB serial whenever an eye blink is
detected.  The Arduino then rotates the stepper motor one quarter turn.

Setup:
  1. Pair the Muse 2 in Bluetooth settings
  2. Terminal A:  muselsl stream
  3. Terminal B:  python blink_to_stepper.py
  4. Blink your eyes -> motor spins
"""

import sys
import time
import numpy as np
from collections import deque

try:
    from pylsl import StreamInlet, resolve_byprop
except ImportError:
    sys.exit("Missing pylsl. Run:  pip install pylsl")

try:
    import serial
    import serial.tools.list_ports
except ImportError:
    sys.exit("Missing pyserial. Run:  pip install pyserial")

# --- Config -------------------------------------------------------------------
SAMPLE_RATE   = 256        # Muse 2 EEG sample rate (Hz)
WINDOW_SIZE   = 50         # ~200 ms sliding window
BAUD          = 115200
NO_DATA_TIMEOUT = 20       # Exit if Muse goes quiet for this many seconds

BLINK_THRESHOLD    = 100   # Spike amplitude in microvolts to count as a blink
MIN_BLINK_INTERVAL = 0.5   # Seconds between accepted blinks (prevents motor spam)
# ------------------------------------------------------------------------------

# Find Arduino -----------------------------------------------------------------
print("Looking for Arduino...")
ports = serial.tools.list_ports.comports()
port = None
for p in ports:
    desc = (p.description + " " + (p.manufacturer or "")).lower()
    if any(k in desc for k in ("arduino", "ch340", "cp210", "ftdi", "usbmodem", "usbserial")):
        port = p.device
        break

if port is None:
    if ports:
        for p in ports:
            print(f"  {p.device}  ({p.description})")
        port = input("Enter Arduino port: ").strip()
    else:
        sys.exit("No serial ports found!")

try:
    ser = serial.Serial(port, BAUD, timeout=1)
except serial.SerialException as e:
    sys.exit(f"Cannot open {port}: {e}")

time.sleep(1)
while ser.in_waiting:
    line = ser.readline().decode(errors="ignore").strip()
    if line:
        print(f"  Arduino: {line}")
print(f"  Connected on {port}")

# Find Muse EEG stream ---------------------------------------------------------
print("\nLooking for Muse EEG stream...")
print("  (make sure 'muselsl stream' is running in another terminal)")
streams = resolve_byprop("type", "EEG", timeout=10)
if not streams:
    ser.close()
    sys.exit("No EEG stream found. Run 'muselsl stream' first.")

inlet = StreamInlet(streams[0])
n_channels = inlet.info().channel_count()
print(f"  Connected: {streams[0].name()}")

# Main loop --------------------------------------------------------------------
print("\nReady - blink to spin the motor (Ctrl+C to quit)\n")

# Muse channel order: TP9 (0), AF7 (1), AF8 (2), TP10 (3)
# Frontal channels AF7 and AF8 pick up eye movement artifacts best.
buffers = [deque(maxlen=WINDOW_SIZE) for _ in range(n_channels)]
last_blink_time = 0
blink_count = 0
samples = 0
no_data_since = time.time()

try:
    while True:
        sample, _ = inlet.pull_sample(timeout=0.5)

        if sample is None:
            print("\rWaiting for Muse data...", end="", flush=True)
            if time.time() - no_data_since > NO_DATA_TIMEOUT:
                ser.close()
                sys.exit("\nNo data for 20 s — is the headband on?")
            continue
        no_data_since = time.time()

        samples += 1
        for i, v in enumerate(sample):
            buffers[i].append(v)

        if len(buffers[0]) < WINDOW_SIZE:
            pct = len(buffers[0]) * 100 // WINDOW_SIZE
            print(f"\rBuffering {pct}%...", end="", flush=True)
            continue

        now = time.time()
        blink_detected = False

        for ch in [1, 2]:  # AF7, AF8
            eeg = np.array(buffers[ch])
            # A blink creates a large, sharp amplitude swing in the frontal
            # electrodes.  We check both peak amplitude and overall variance
            # to avoid triggering on slow baseline drift.
            if np.max(np.abs(eeg)) > BLINK_THRESHOLD and np.std(eeg) > 30:
                if now - last_blink_time > MIN_BLINK_INTERVAL:
                    blink_detected = True
                    break

        if blink_detected:
            ser.write(b"BLINK\n")
            last_blink_time = now
            blink_count += 1
            print(f"BLINK #{blink_count} -> motor spinning")
            time.sleep(0.1)
            while ser.in_waiting:
                line = ser.readline().decode(errors="ignore").strip()
                if line:
                    print(f"  Arduino: {line}")
        else:
            print(f"\r  Monitoring | blinks: {blink_count} | samples: {samples}     ",
                  end="", flush=True)

except KeyboardInterrupt:
    print(f"\n\nStopped. Total blinks detected: {blink_count}")
    ser.close()
