/*
 * Blink-Controlled Stepper Motor
 * ================================
 * Receives "BLINK" commands over USB serial and rotates a 28BYJ-48
 * stepper motor 90 degrees each time. A button on pin 2 toggles
 * detection on/off, so you can pause the motor without stopping the
 * Python side.
 *
 * Wiring (ULN2003 driver board):
 *   IN1 -> Arduino pin 8
 *   IN2 -> Arduino pin 9
 *   IN3 -> Arduino pin 10
 *   IN4 -> Arduino pin 11
 *   +   -> 5V
 *   -   -> GND
 *   Button (one leg) -> pin 2, other leg -> GND
 */

#include <Arduino.h>
#include <Stepper.h>

#define STEPS_PER_REV 2048   // 28BYJ-48 full revolution in half-step mode
#define ROTATION_STEPS 512   // quarter turn (90 degrees)
#define BUTTON_PIN 2
#define DEBOUNCE_MS 40

// Note: ULN2003 wiring interleaves coil pins -> pass as 8,10,9,11
Stepper motor(STEPS_PER_REV, 8, 10, 9, 11);

String serialBuffer = "";
int blinkCount = 0;
bool detectionActive = false;
bool lastButtonReading = HIGH;
bool stableButtonState = HIGH;
unsigned long lastDebounceChange = 0;

void setup() {
    Serial.begin(115200);
    motor.setSpeed(10);  // 10 RPM

    pinMode(LED_BUILTIN, OUTPUT);
    pinMode(BUTTON_PIN, INPUT_PULLUP);
    digitalWrite(LED_BUILTIN, LOW);

    Serial.println("=================================");
    Serial.println("BLINK STEPPER MOTOR CONTROLLER");
    Serial.println("=================================");
    Serial.println("Press button on pin 2 to START/PAUSE");
    Serial.println();
    Serial.println("PAUSED");
}

void loop() {
    // Debounce the button (active low - button pulls pin to GND when pressed)
    bool reading = digitalRead(BUTTON_PIN);
    if (reading != lastButtonReading) {
        lastDebounceChange = millis();
        lastButtonReading = reading;
    }
    if ((millis() - lastDebounceChange) > DEBOUNCE_MS && reading != stableButtonState) {
        stableButtonState = reading;
        if (stableButtonState == LOW) {
            detectionActive = !detectionActive;
            digitalWrite(LED_BUILTIN, detectionActive ? HIGH : LOW);
            // Python script reads this to know whether to send blinks
            Serial.println(detectionActive ? "ACTIVE" : "PAUSED");
        }
    }

    // Read incoming serial data character by character
    while (Serial.available()) {
        char c = Serial.read();
        if (c == '\n') {
            serialBuffer.trim();

            if (serialBuffer == "BLINK") {
                if (!detectionActive) {
                    serialBuffer = "";
                    continue;
                }
                blinkCount++;
                Serial.print("BLINK #");
                Serial.print(blinkCount);
                Serial.println(" - spinning motor");

                // Turn off the indicator LED while the motor runs
                // (the motor blocks, so it gives visual feedback)
                digitalWrite(LED_BUILTIN, LOW);
                motor.step(ROTATION_STEPS);
                digitalWrite(LED_BUILTIN, detectionActive ? HIGH : LOW);

                Serial.println("Done");
            }

            serialBuffer = "";
        } else {
            serialBuffer += c;
        }
    }
}
