#!/usr/bin/env bash
set -euo pipefail

PROJECT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"
cd "$PROJECT_DIR"

BIN_NAME="${1:-grove_d16}"
TTY_PORT="${PICO_TTY_PORT:-/dev/ttyS0}"
ELF_PATH="target/thumbv6m-none-eabi/release/${BIN_NAME}"
UF2_PATH="${BIN_NAME}.uf2"
PICOTOOL="${HOME}/.platformio/packages/tool-rp2040tools/picotool"
BOOTSEL_MOUNT="/media/${USER}/RPI-RP2"

echo "Building ${BIN_NAME}..."
cargo build --release --offline --bin "${BIN_NAME}"

echo "Creating ${UF2_PATH}..."
elf2uf2-rs "${ELF_PATH}" "${UF2_PATH}"

if [[ -e "${TTY_PORT}" ]]; then
  echo "Attempting 1200-bps reset on ${TTY_PORT}..."
  stty -F "${TTY_PORT}" 1200 || true
  sleep 0.3
fi

if [[ -x "${PICOTOOL}" ]]; then
  echo "Uploading with picotool..."
  if ! "${PICOTOOL}" load "${UF2_PATH}" -t uf2 -x; then
    if [[ -d "${BOOTSEL_MOUNT}" ]]; then
      echo "picotool failed, but BOOTSEL drive is mounted; copying UF2 directly..."
      cp "${UF2_PATH}" "${BOOTSEL_MOUNT}/"
      sync
    else
      echo "No RP2040 in BOOTSEL mode found."
      echo "Press and hold BOOTSEL while plugging in USB, then run again."
      exit 1
    fi
  fi
  echo "Done."
else
  echo "picotool not found at ${PICOTOOL}."
  echo "Copy ${UF2_PATH} to RPI-RP2 manually in BOOTSEL mode."
  exit 1
fi
