// ======================================================
//  Locus Pocus Secondary - Clock Device / Motor Control
// ======================================================
/* 

ClockHand.cpp - 
Class representing Locus Pocus clock hand as part of the clock device, 
providing for setup and management of a clock hand and associated 
movement driven by a stepper motor in response to clock commands.  

Copyright (C) 2025 - David Wilson

This program is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <https://www.gnu.org/licenses/>.

*/

#include "ClockHand.h"

ClockHand::ClockHand(std::string name, motor_t motor, bool motorOrientation, uint8_t pin1, uint8_t pin2, uint8_t pin3, uint8_t pin4)
  : AccelStepper(AccelStepper::FULL4WIRE, pin1, pin2, pin3, pin4) {
  this->setMaxSpeed(1000.0);
  this->setAcceleration(100.0);
  this->setSpeed(1000);
  this->name = name;
  _motor = motor;
  _motorOrientation = motorOrientation;
}

void ClockHand::moveTo(long motorPositionInSteps) {
  AccelStepper::moveTo((_motorOrientation ? 1 : -1) * motorPositionInSteps);
}

void ClockHand::moveToDegree(int positionInDegrees) {
  double degreeToStepScale = static_cast<double>(positionInDegrees) / 360.0;
  double motorPositionInSteps = degreeToStepScale * _motor.stepsPerRevolution;
  this->moveTo(static_cast<long>(motorPositionInSteps));  
}

void ClockHand::calibrate(Stream& io) {
  io.print("Calibrating: ");
  io.print(this->name.c_str());
  io.println(" [Enter] to set base position (max 2 rotations to try)...");
  io.flush();
  delay(1000);
  this->moveTo(AccelStepper::currentPosition() + 2*_motor.stepsPerRevolution);
  while (AccelStepper::distanceToGo() != 0)
  {
    AccelStepper::run();
    if (io.available())
    {
      AccelStepper::setCurrentPosition(0);
      while (io.available()) {
        Serial.read();
      }
    }
  }
}
