#ifndef OLED_I2C_H
#define OLED_I2C_H

#include "ch32fun.h"
#include <stdio.h>

// ===================================================================================
// I2C Bit-Bang Driver
// ===================================================================================

// User defined pins for Soft I2C
#define I2C_SDA PA6
#define I2C_SCL PA5

// Simple delay for bit-banging
static void i2c_delay(void) {
  for (volatile int i = 0; i < 2; i++) {
    __asm__("nop");
  }
}

static void i2c_sda_high(void) {
  // Set as Input with Pull-up
  funPinMode(I2C_SDA, GPIO_CFGLR_IN_PUPD);
  funDigitalWrite(I2C_SDA, 1);
}

static void i2c_sda_low(void) {
  // Set as Output Low
  funPinMode(I2C_SDA, GPIO_CFGLR_OUT_10Mhz_PP);
  funDigitalWrite(I2C_SDA, 0);
}

static void i2c_scl_high(void) { funDigitalWrite(I2C_SCL, 1); }

static void i2c_scl_low(void) { funDigitalWrite(I2C_SCL, 0); }

static void i2c_start(void) {
  i2c_sda_high();
  i2c_scl_high();
  i2c_delay();
  i2c_sda_low();
  i2c_delay();
  i2c_scl_low();
}

static void i2c_stop(void) {
  i2c_sda_low();
  i2c_delay();
  i2c_scl_high();
  i2c_delay();
  i2c_sda_high();
  i2c_delay();
}

static u8 i2c_write_byte(u8 byte) {
  for (int i = 0; i < 8; i++) {
    if (byte & 0x80)
      i2c_sda_high();
    else
      i2c_sda_low();
    byte <<= 1;
    i2c_delay();
    i2c_scl_high();
    i2c_delay();
    i2c_scl_low();
  }

  // ACKnowledge Pulse
  i2c_sda_high(); // Release SDA
  i2c_delay();
  i2c_scl_high();
  i2c_delay();

  // Sample SDA
  u8 ack = funDigitalRead(I2C_SDA);

  i2c_scl_low();
  return ack;
}

static u8 i2c_writeData(u8 address, u8 *data, u8 len) {
  i2c_start();

  // Send Address + Write(0)
  if (i2c_write_byte(address << 1)) {
    i2c_stop();
    return 1; // NACK
  }

  for (u8 i = 0; i < len; i++) {
    if (i2c_write_byte(data[i])) {
      i2c_stop();
      return 2; // NACK
    }
  }

  i2c_stop();
  return 0; // Success
}

static u8 i2c_init(void) {
  // SCL Output Push-Pull
  funPinMode(I2C_SCL, GPIO_CFGLR_OUT_10Mhz_PP);
  i2c_scl_high();

  // SDA Init as High (Input)
  i2c_sda_high();

  return 0;
}

// ===================================================================================
// SSD1306 OLED Driver
// ===================================================================================

#define SSD1306_ADDRESS 0x3C

static const uint8_t font5x7[] = {
    0x00, 0x00, 0x00, 0x00, 0x00, // space
    0x00, 0x00, 0x5F, 0x00, 0x00, // !
    0x00, 0x07, 0x00, 0x07, 0x00, // "
    0x14, 0x7F, 0x14, 0x7F, 0x14, // #
    0x24, 0x2A, 0x7F, 0x2A, 0x12, // $
    0x23, 0x13, 0x08, 0x64, 0x62, // %
    0x36, 0x49, 0x55, 0x22, 0x50, // &
    0x00, 0x05, 0x03, 0x00, 0x00, // '
    0x00, 0x1C, 0x22, 0x41, 0x00, // (
    0x00, 0x41, 0x22, 0x1C, 0x00, // )
    0x14, 0x08, 0x3E, 0x08, 0x14, // *
    0x08, 0x08, 0x3E, 0x08, 0x08, // +
    0x00, 0x50, 0x60, 0x00, 0x00, // ,
    0x08, 0x08, 0x08, 0x08, 0x08, // -
    0x00, 0x60, 0x60, 0x00, 0x00, // .
    0x20, 0x10, 0x08, 0x04, 0x02, // /
    0x3E, 0x51, 0x49, 0x45, 0x3E, // 0
    0x00, 0x42, 0x7F, 0x40, 0x00, // 1
    0x42, 0x61, 0x51, 0x49, 0x46, // 2
    0x21, 0x41, 0x45, 0x4B, 0x31, // 3
    0x18, 0x14, 0x12, 0x7F, 0x10, // 4
    0x27, 0x45, 0x45, 0x45, 0x39, // 5
    0x3C, 0x4A, 0x49, 0x49, 0x30, // 6
    0x01, 0x71, 0x09, 0x05, 0x03, // 7
    0x36, 0x49, 0x49, 0x49, 0x36, // 8
    0x06, 0x49, 0x49, 0x29, 0x1E, // 9
    0x00, 0x36, 0x36, 0x00, 0x00, // :
    0x00, 0x56, 0x36, 0x00, 0x00, // ;
    0x08, 0x14, 0x22, 0x41, 0x00, // <
    0x14, 0x14, 0x14, 0x14, 0x14, // =
    0x00, 0x41, 0x22, 0x14, 0x08, // >
    0x02, 0x01, 0x51, 0x09, 0x06, // ?
    0x32, 0x49, 0x79, 0x41, 0x3E, // @
    0x7E, 0x11, 0x11, 0x11, 0x7E, // A
    0x7F, 0x49, 0x49, 0x49, 0x36, // B
    0x3E, 0x41, 0x41, 0x41, 0x22, // C
    0x7F, 0x41, 0x41, 0x22, 0x1C, // D
    0x7F, 0x49, 0x49, 0x49, 0x41, // E
    0x7F, 0x09, 0x09, 0x09, 0x01, // F
    0x3E, 0x41, 0x49, 0x49, 0x7A, // G
    0x7F, 0x08, 0x08, 0x08, 0x7F, // H
    0x00, 0x41, 0x7F, 0x41, 0x00, // I
    0x20, 0x40, 0x41, 0x3F, 0x01, // J
    0x7F, 0x08, 0x14, 0x22, 0x41, // K
    0x7F, 0x40, 0x40, 0x40, 0x40, // L
    0x7F, 0x02, 0x0C, 0x02, 0x7F, // M
    0x7F, 0x04, 0x08, 0x10, 0x7F, // N
    0x3E, 0x41, 0x41, 0x41, 0x3E, // O
    0x7F, 0x09, 0x09, 0x09, 0x06, // P
    0x3E, 0x41, 0x51, 0x21, 0x5E, // Q
    0x7F, 0x09, 0x19, 0x29, 0x46, // R
    0x46, 0x49, 0x49, 0x49, 0x31, // S
    0x01, 0x01, 0x7F, 0x01, 0x01, // T
    0x3F, 0x40, 0x40, 0x40, 0x3F, // U
    0x1F, 0x20, 0x40, 0x20, 0x1F, // V
    0x3F, 0x40, 0x38, 0x40, 0x3F, // W
    0x63, 0x14, 0x08, 0x14, 0x63, // X
    0x07, 0x08, 0x70, 0x08, 0x07, // Y
    0x61, 0x51, 0x49, 0x45, 0x43, // Z
    0x00, 0x7F, 0x41, 0x41, 0x00, // [
    0x02, 0x04, 0x08, 0x10, 0x20, // "\"
    0x00, 0x41, 0x41, 0x7F, 0x00, // ]
    0x04, 0x02, 0x01, 0x02, 0x04, // ^
    0x40, 0x40, 0x40, 0x40, 0x40, // _
    0x00, 0x01, 0x02, 0x04, 0x00, // `
    0x20, 0x54, 0x54, 0x54, 0x78, // a
    0x7F, 0x48, 0x44, 0x44, 0x38, // b
    0x38, 0x44, 0x44, 0x44, 0x20, // c
    0x38, 0x44, 0x44, 0x48, 0x7F, // d
    0x38, 0x54, 0x54, 0x54, 0x18, // e
    0x08, 0x7E, 0x09, 0x01, 0x02, // f
    0x0C, 0x52, 0x52, 0x52, 0x3E, // g
    0x7F, 0x08, 0x04, 0x04, 0x78, // h
    0x00, 0x44, 0x7D, 0x40, 0x00, // i
    0x20, 0x40, 0x44, 0x3D, 0x00, // j
    0x7F, 0x10, 0x28, 0x44, 0x00, // k
    0x00, 0x41, 0x7F, 0x40, 0x00, // l
    0x7C, 0x04, 0x18, 0x04, 0x78, // m
    0x7C, 0x08, 0x04, 0x04, 0x78, // n
    0x38, 0x44, 0x44, 0x44, 0x38, // o
    0x7C, 0x14, 0x14, 0x14, 0x08, // p
    0x08, 0x14, 0x14, 0x18, 0x7C, // q
    0x7C, 0x08, 0x04, 0x04, 0x08, // r
    0x48, 0x54, 0x54, 0x54, 0x20, // s
    0x04, 0x3F, 0x44, 0x40, 0x20, // t
    0x3C, 0x40, 0x40, 0x20, 0x7C, // u
    0x1C, 0x20, 0x40, 0x20, 0x1C, // v
    0x3C, 0x40, 0x30, 0x40, 0x3C, // w
    0x44, 0x28, 0x10, 0x28, 0x44, // x
    0x0C, 0x50, 0x50, 0x50, 0x3C, // y
    0x44, 0x64, 0x54, 0x4C, 0x44, // z
    0x00, 0x08, 0x36, 0x41, 0x00, // {
    0x00, 0x00, 0x7F, 0x00, 0x00, // |
    0x00, 0x41, 0x36, 0x08, 0x00, // }
    0x10, 0x08, 0x08, 0x10, 0x08, // ~
};

static void ssd1306_cmd(uint8_t cmd) {
  uint8_t data[2] = {0x00, cmd};
  i2c_writeData(SSD1306_ADDRESS, data, 2);
}

static void ssd1306_data(uint8_t d) {
  uint8_t data[2] = {0x40, d};
  i2c_writeData(SSD1306_ADDRESS, data, 2);
}

static void ssd1306_init(void) {
  // Init sequence for 128x32
  uint8_t init_sequence[] = {
      0xAE,       // Display Off
      0xD5, 0x80, // Set Display Clock Divide Ratio
      0xA8, 0x1F, // Set Multiplex Ratio (32 lines)
      0xD3, 0x00, // Set Display Offset
      0x40,       // Set Start Line 0
      0x8D, 0x14, // Charge Pump Enable
      0x20, 0x00, // Memory Addressing Mode (Horizontal)
      0xA1,       // Segment Re-map
      0xC8,       // COM Output Scan Direction
      0xDA, 0x02, // COM Pins Hardware Config
      0x81, 0x8F, // Set Contrast
      0xD9, 0xF1, // Set Pre-charge Period
      0xDB, 0x40, // Set VCOMH Deselect Level
      0xA4,       // Resume from RAM content
      0xA6,       // Normal Display
      0xAF        // Display On
  };

  i2c_start();
  i2c_write_byte(SSD1306_ADDRESS << 1);
  i2c_write_byte(0x00); // Command Stream (Co=0, D/C#=0)

  for (int i = 0; i < sizeof(init_sequence); i++) {
    i2c_write_byte(init_sequence[i]);
  }

  i2c_stop();
}

static void ssd1306_clear(void) {
  for (uint8_t page = 0; page < 4; page++) {
    ssd1306_cmd(0xB0 + page);
    ssd1306_cmd(0x00);
    ssd1306_cmd(0x10);

    i2c_start();
    i2c_write_byte(SSD1306_ADDRESS << 1);
    i2c_write_byte(0x40); // Data Stream
    for (uint8_t col = 0; col < 128; col++) {
      i2c_write_byte(0x00);
    }
    i2c_stop();
  }
}

static void ssd1306_set_cursor(uint8_t page, uint8_t col) {
  ssd1306_cmd(0xB0 + page);
  ssd1306_cmd(0x00 | (col & 0x0F));
  ssd1306_cmd(0x10 | ((col >> 4) & 0x0F));
}

static void ssd1306_write_char(char c) {
  if (c < 32 || c > 126)
    return;
  const uint8_t *bitmap = &font5x7[(c - 32) * 5];

  i2c_start();
  i2c_write_byte(SSD1306_ADDRESS << 1);
  i2c_write_byte(0x40); // Data Stream
  for (uint8_t i = 0; i < 5; i++) {
    i2c_write_byte(bitmap[i]);
  }
  i2c_write_byte(0x00); // Space
  i2c_stop();
}

static void ssd1306_print(const char *str) {
  while (*str) {
    ssd1306_write_char(*str++);
  }
}

#endif
