#include <Wire.h>
#include <WiFi.h>
#include <Adafruit_PWMServoDriver.h>

// ======================
// ==== WiFi Settings ====
// ======================
const char* ssid = "*******";
const char* password = "******";
WiFiServer server(80);

// ============================
// ==== Servo Driver Setup ====
// ============================
Adafruit_PWMServoDriver pwm = Adafruit_PWMServoDriver();
#define SERVO_FREQ 50  // Hz

// === Servo Channels ===
// Front Left: 0–1
// Front Right: 2–3
// Back Left: 4–5
// Back Right: 6–7
#define FL_HIP 0
#define FL_KNEE 1
#define FR_HIP 2
#define FR_KNEE 3
#define BL_HIP 4
#define BL_KNEE 5
#define BR_HIP 6
#define BR_KNEE 7

// === Servo Pulse Ranges ===
#define HIP_MIN  350
#define HIP_MAX  370
#define KNEE_MIN 340
#define KNEE_MAX 420

// === Control ===
volatile bool stopAll = false;
int command = 0;

// =======================
// ==== Helper Functions ====
// =======================
void moveServo(uint8_t servo, uint16_t start, uint16_t end, uint8_t stepDelay = 5) {
  if (stopAll) return;
  if (start < end) {
    for (uint16_t pos = start; pos <= end; pos++) {
      if (stopAll) return;
      pwm.setPWM(servo, 0, pos);
      delay(stepDelay);
    }
  } else {
    for (uint16_t pos = start; pos >= end; pos--) {
      if (stopAll) return;
      pwm.setPWM(servo, 0, pos);
      delay(stepDelay);
      if (pos == end) break; // Prevent underflow
    }
  }
}

void homeLeg(uint8_t hip, uint8_t knee) {
  pwm.setPWM(hip, 0, HIP_MIN);
  pwm.setPWM(knee, 0, KNEE_MIN);
}

void homeAll() {
  stopAll = true;  // stop everything immediately
  delay(50);
  stopAll = false;
  homeLeg(FL_HIP, FL_KNEE);
  homeLeg(FR_HIP, FR_KNEE);
  homeLeg(BL_HIP, BL_KNEE);
  homeLeg(BR_HIP, BR_KNEE);
}

// =======================
// ==== Motion Patterns ====
// =======================

// Smooth continuous diagonal gait
void walkForward() {
  stopAll = false;

  uint16_t HIP_FWD = HIP_MIN + 8;
  uint16_t HIP_BWD = HIP_MAX - 8;
  uint16_t KNEE_UP = KNEE_MIN + 10;
  uint16_t KNEE_DOWN = KNEE_MAX - 10;

  // Initial stance
  pwm.setPWM(FL_HIP, 0, HIP_MIN);
  pwm.setPWM(FR_HIP, 0, HIP_MIN);
  pwm.setPWM(BL_HIP, 0, HIP_MIN);
  pwm.setPWM(BR_HIP, 0, HIP_MIN);

  pwm.setPWM(FL_KNEE, 0, KNEE_MIN);
  pwm.setPWM(FR_KNEE, 0, KNEE_MIN);
  pwm.setPWM(BL_KNEE, 0, KNEE_MIN);
  pwm.setPWM(BR_KNEE, 0, KNEE_MIN);

  // Continuous walking loop until stopped
  while (!stopAll) {
    // Step phase 1
    for (uint16_t pos = HIP_MIN; pos <= HIP_MAX; pos++) {
      if (stopAll) return;

      uint16_t opposite = HIP_MAX - (pos - HIP_MIN);
      pwm.setPWM(FL_HIP, 0, pos);
      pwm.setPWM(BR_HIP, 0, pos);
      pwm.setPWM(FR_HIP, 0, opposite);
      pwm.setPWM(BL_HIP, 0, opposite);

      pwm.setPWM(FL_KNEE, 0, map(pos, HIP_MIN, HIP_MAX, KNEE_MIN, KNEE_UP));
      pwm.setPWM(BR_KNEE, 0, map(pos, HIP_MIN, HIP_MAX, KNEE_MIN, KNEE_UP));
      pwm.setPWM(FR_KNEE, 0, map(opposite, HIP_MIN, HIP_MAX, KNEE_UP, KNEE_MIN));
      pwm.setPWM(BL_KNEE, 0, map(opposite, HIP_MIN, HIP_MAX, KNEE_UP, KNEE_MIN));

      delay(10);
    }

    // Step phase 2
    for (uint16_t pos = HIP_MAX; pos >= HIP_MIN; pos--) {
      if (stopAll) return;

      uint16_t opposite = HIP_MAX - (pos - HIP_MIN);
      pwm.setPWM(FL_HIP, 0, pos);
      pwm.setPWM(BR_HIP, 0, pos);
      pwm.setPWM(FR_HIP, 0, opposite);
      pwm.setPWM(BL_HIP, 0, opposite);

      pwm.setPWM(FL_KNEE, 0, map(pos, HIP_MIN, HIP_MAX, KNEE_MIN, KNEE_UP));
      pwm.setPWM(BR_KNEE, 0, map(pos, HIP_MIN, HIP_MAX, KNEE_MIN, KNEE_UP));
      pwm.setPWM(FR_KNEE, 0, map(opposite, HIP_MIN, HIP_MAX, KNEE_UP, KNEE_MIN));
      pwm.setPWM(BL_KNEE, 0, map(opposite, HIP_MIN, HIP_MAX, KNEE_UP, KNEE_MIN));

      delay(10);
      if (pos == HIP_MIN) break;
    }
  }

  homeAll();
}

// Sit: fold knees, lower hips
void sitPose() {
  stopAll = true; delay(50); stopAll = false;
  moveServo(FL_KNEE, KNEE_MIN, KNEE_MAX, 5);
  moveServo(FR_KNEE, KNEE_MIN, KNEE_MAX, 5);
  moveServo(BL_KNEE, KNEE_MIN, KNEE_MAX, 5);
  moveServo(BR_KNEE, KNEE_MIN, KNEE_MAX, 5);

  moveServo(FL_HIP, HIP_MIN, HIP_MAX, 5);
  moveServo(FR_HIP, HIP_MIN, HIP_MAX, 5);
  moveServo(BL_HIP, HIP_MIN, HIP_MAX, 5);
  moveServo(BR_HIP, HIP_MIN, HIP_MAX, 5);
}

// Dance: 5 seconds of movement
void danceMove() {
  stopAll = false;
  unsigned long startTime = millis();
  while (millis() - startTime < 5000 && !stopAll) {
    moveServo(FL_HIP, HIP_MIN, HIP_MAX, 3);
    moveServo(FR_HIP, HIP_MAX, HIP_MIN, 3);
    moveServo(BL_HIP, HIP_MIN, HIP_MAX, 3);
    moveServo(BR_HIP, HIP_MAX, HIP_MIN, 3);

    moveServo(FL_KNEE, KNEE_MIN, KNEE_MAX, 3);
    moveServo(FR_KNEE, KNEE_MAX, KNEE_MIN, 3);
    moveServo(BL_KNEE, KNEE_MIN, KNEE_MAX, 3);
    moveServo(BR_KNEE, KNEE_MAX, KNEE_MIN, 3);
  }
}

// =======================
// ==== Setup ====
// =======================
void setup() {
  Serial.begin(115200);
  pwm.begin();
  pwm.setOscillatorFrequency(27000000);
  pwm.setPWMFreq(SERVO_FREQ);
  homeAll();

  WiFi.begin(ssid, password);
  Serial.print("Connecting to WiFi");
  while (WiFi.status() != WL_CONNECTED) {
    delay(500);
    Serial.print(".");
  }

  Serial.println("\nConnected!");
  Serial.println(WiFi.localIP());
  server.begin();
}

// =======================
// ==== Web Server ====
// =======================
void handleClient() {
  WiFiClient client = server.available();
  if (!client) return;

  String request = client.readStringUntil('\r');
  client.flush();

  // Reset current command before setting a new one
  stopAll = true;
  delay(50);
  stopAll = false;

  if (request.indexOf("/FORWARD") != -1) command = 1;
  else if (request.indexOf("/SIT") != -1) command = 2;
  else if (request.indexOf("/DANCE") != -1) command = 3;
  else if (request.indexOf("/HOME") != -1) command = 4;

  String html = R"rawliteral(
    <!DOCTYPE html>
    <html>
    <head><title>Robot Dog Control</title></head>
    <body style="background-color:#111; color:white; font-family:sans-serif; text-align:center;">
      <h2>ESP32 Robot Dog</h2>
      <button onclick="location.href='/FORWARD'" style="width:200px; height:60px; margin:10px; background-color:#0077ff;">Forward</button><br>
      <button onclick="location.href='/SIT'" style="width:200px; height:60px; margin:10px; background-color:#ffaa00;">Sit</button><br>
      <button onclick="location.href='/DANCE'" style="width:200px; height:60px; margin:10px; background-color:#aa00ff;">Dance</button><br>
      <button onclick="location.href='/HOME'" style="width:200px; height:60px; margin:10px; background-color:#00cc00;">Home</button>
    </body>
    </html>
  )rawliteral";

  client.println("HTTP/1.1 200 OK");
  client.println("Content-Type: text/html");
  client.println("Connection: close");
  client.println();
  client.print(html);
}

// =======================
// ==== Main Loop ====
// =======================
void loop() {
  handleClient();

  switch (command) {
    case 1: walkForward(); break;
    case 2: sitPose(); break;
    case 3: danceMove(); break;
    case 4: homeAll(); break;
  }

  command = 0;
}
